import email
import importlib.resources as pkg_resources

from fastapi import HTTPException
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit import dateutils
from be_kit.formatters.accountings import format_currency
from be_kit.mailing import EmailClient
from be_kit.paginations import PaginationQuery
from be_procurement.transaction.enums import TransactionStatus
from be_uam.organization.models import Group
from be_uam.user.models import User
from ..settings import Settings
from . import models, schemas, repositories


async def create_requisition(
    db: AsyncSession,
    requisition: schemas.RequisitionCreate,
    request_user: User,
    _simulate: bool = False,
):
    requisition_data = requisition.model_dump()
    requisition_items = requisition_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.RequisitionItemRepository(
        db,
        eager_loads=[
            "product",
            "vat",
            "vat.country",
        ],
    )
    obj = await repo.acreate(
        **requisition_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=not _simulate,
    )
    for item_data in requisition_items:
        item_data["requisition_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def retrieve_requisition(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_requisitions(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.RequisitionFilter,
    request_user: User,
    as_options: bool = False,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, as_options)
    return objs


async def update_requisition(
    db: AsyncSession,
    pk: int,
    requisition: schemas.RequisitionUpdate,
    request_user: User,
    _simulate: bool = False,
):
    requisition_data = requisition.model_dump()
    requisition_items = requisition_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.RequisitionItemRepository(db)
    obj = await repo.aget_or_404(pk)
    if obj.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Only draft/issued requisition can be updated")
    obj = await repo.aupdate(
        pk,
        **requisition_data,
        status=TransactionStatus.DRAFT,
        last_modified_by_id=request_user.pk,
        _commit=not _simulate,
    )
    await item_repo.adelete_by_requisition_id(pk)
    for item_data in requisition_items:
        item_data["requisition_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def delete_requisition(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def send_requisition_email(
    email: EmailStr,
    requisition: models.Requisition,
    request_user: User,
    settings: Settings,
):
    if requisition.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Requisition is not draft/issued")
    group: Group = await request_user.awaitable_attrs.group
    email_client = EmailClient(
        settings=settings.mailing,
        template_dir=pkg_resources.files("symbolix_procurement.templates").joinpath(
            "email"
        ),
        test_mode=settings.test_mode,
        debug=settings.debug,
    )
    email_client.template_env.globals["format_currency"] = format_currency
    message = await email_client.get_templated_message(
        template_name="requisition.html",
        subject=f"{group.organization.legal_name} Requisition #{requisition.requisition_id}",
        to=email,
        body={
            "requisition_id": requisition.requisition_id,
            "email": email,
            "organization_name": group.organization.legal_name,
            "items": requisition.items,
            "submitted_by": request_user.first_name + " " + request_user.last_name,
            "department": group.name,
            "total": requisition.total,
            "total_discount": requisition.total_discount,
            "total_vat": requisition.total_vat,
            "total_net": requisition.total_net,
            "notes": requisition.notes or "-",
            "year": dateutils.now().year,
            "currency": requisition.currency.symbol,
        },
    )
    await email_client.send_message(message)


async def preview_requisition_email(
    requisition: models.Requisition,
    request_user: User,
    settings: Settings,
):
    group: Group = await request_user.awaitable_attrs.group
    email_client = EmailClient(
        settings=settings.mailing,
        template_dir=pkg_resources.files("symbolix_procurement.templates").joinpath(
            "email"
        ),
        test_mode=settings.test_mode,
        debug=settings.debug,
    )
    email_client.template_env.globals["format_currency"] = format_currency
    preview = await email_client.get_html_preview(
        template_name="requisition.html",
        body={
            "requisition_id": requisition.requisition_id,
            "email": email,
            "organization_name": group.organization.legal_name,
            "items": requisition.items,
            "submitted_by": request_user.first_name + " " + request_user.last_name,
            "department": group.name,
            "total": requisition.total,
            "total_discount": requisition.total_discount,
            "total_vat": requisition.total_vat,
            "total_net": requisition.total_net,
            "notes": requisition.notes or "-",
            "year": dateutils.now().year,
            "currency": requisition.currency.symbol,
        },
    )
    return preview


async def set_requisition_to_issued(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Requisition is not in draft/issued status")
    if obj.status == TransactionStatus.DRAFT:
        await repo.aupdate(
            pk, status=TransactionStatus.ISSUED, last_modified_by_id=request_user.pk
        )


async def approve_requisition(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aupdate(
        pk, status=TransactionStatus.APPROVED, approved_by_id=request_user.pk
    )
    return obj


async def cancel_requisition(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.RequisitionRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status in [TransactionStatus.COMPLETED, TransactionStatus.CANCELLED]:
        raise HTTPException(400, "Cannot cancel completed requisition")
    obj = await repo.aupdate(
        pk, status=TransactionStatus.CANCELLED, last_modified_by_id=request_user.pk
    )
    return obj
