from datetime import date
from sqlalchemy import BigInteger, Date, ForeignKey, String, UniqueConstraint
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import (
    BaseModel,
    SoftDeleteMixin,
)
from be_uam.kit.models import MetadataMixin
from be_uam.organization.models import OrganizationMixin

from be_procurement.transaction.enums import (
    TransactionStatus,
    TransactionPaymentTerm,
    TransactionDeliveryTerm,
)
from be_procurement.transaction.models import (
    TransactionMixin,
    TransactionItemMixin,
)
from be_core_services.country.models import VAT, Currency
from be_inventory.product.models import Product
from ..vendor.models import Vendor


class Requisition(
    TransactionMixin, OrganizationMixin, SoftDeleteMixin, MetadataMixin, BaseModel
):
    __tablename__ = "procurement_requisition"

    requisition_id: Mapped[str] = mapped_column(String(11))
    vendor_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Vendor.pk, ondelete="RESTRICT")
    )
    vendor: Mapped[Vendor] = relationship(
        "Vendor",
        passive_deletes=True,
        lazy="selectin",
    )
    items: Mapped[list["RequisitionItem"]] = relationship(
        "RequisitionItem",
        back_populates="requisition",
        cascade="all, delete",
        passive_deletes=True,
        lazy="selectin",
    )


class RequisitionCounter(OrganizationMixin, BaseModel):
    __tablename__ = "procurement_requisition_counter"
    __table_args__ = (
        UniqueConstraint(
            "organization_id",
            "counter_date",
            name=f"uq_{__tablename__}_organization_id_counter_date",
        ),
    )

    counter_date: Mapped[date] = mapped_column(Date())
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class RequisitionItem(TransactionItemMixin, BaseModel):
    __tablename__ = "procurement_requisition_item"
    __table_args__ = (
        UniqueConstraint(
            "requisition_id",
            "product_id",
            name=f"uq_{__tablename__}_requisition_id_product_id",
        ),
    )

    requisition_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Requisition.pk, ondelete="CASCADE")
    )
    requisition: Mapped[Requisition] = relationship(
        "Requisition",
        back_populates="items",
        lazy="selectin",
    )
