from typing import Annotated
from fastapi import APIRouter, BackgroundTasks, Depends, status
from fastapi.params import Query
from fastapi.responses import HTMLResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from . import schemas, utils, enums
from .permissions import requisition_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["requisitions"], prefix="/requisition")


@router.post(
    "/",
    dependencies=[Depends(authorize(requisition_permission_mapper["create_requisition"]))],
    response_model=schemas.Requisition,
)
async def create_requisition(
    requisition: schemas.RequisitionCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_requisition(session, requisition, request_user)


@router.post(
    "/simulate/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["create_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def simulate_create_requisition(
    requisition: schemas.RequisitionCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_requisition(session, requisition, request_user, _simulate=True)


@router.get(
    "/{requisition_id:int}/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["retrieve_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def retrieve_requisition(
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_requisition(session, requisition_id, request_user)


@router.get(
    "/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["list_requisition"]))
    ],
    response_model=schemas.PaginatedRequisition,
)
async def list_requisition(
    pagination: PaginationQuery = Depends(),
    filters: schemas.RequisitionFilter = Depends(),
    ordering: Annotated[list[enums.RequisitionOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_requisitions(session, pagination, filters, request_user, ordering)

@router.get(
    "/options/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["list_requisition_option"]))
    ],
    response_model=schemas.PaginatedRequisitionOpt,
)
async def list_requisition_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.RequisitionFilter = Depends(),
    ordering: Annotated[list[enums.RequisitionOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_requisitions(session, pagination, filters, request_user, ordering)


@router.put(
    "/{requisition_id:int}/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["update_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def update_requisition(
    requisition: schemas.RequisitionUpdate,
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_requisition(
        session, requisition_id, requisition, request_user
    )


@router.put(
    "/simulate/{requisition_id:int}/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["update_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def simulate_update_requisition(
    requisition: schemas.RequisitionUpdate,
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_requisition(
        session, requisition_id, requisition, request_user, _simulate=True
    )


@router.delete(
    "/{requisition_id:int}/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["delete_requisition"]))
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_requisition(
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_requisition(session, requisition_id, request_user)


@router.post(
    "/{requisition_id:int}/approve/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["approve_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def approve_requisition(
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_requisition(session, requisition_id, request_user)


@router.post(
    "/{requisition_id:int}/send_email/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["send_requisition_email"]))
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def send_requisition_email(
    requisition_id: int,
    body: schemas.RequisitionEmailSend,
    background_tasks: BackgroundTasks,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    obj = await utils.retrieve_requisition(session, requisition_id, request_user)
    background_tasks.add_task(
        utils.send_requisition_email,
        body.email,
        obj,
        request_user,
    )
    await utils.set_requisition_to_issued(session, requisition_id, request_user)


@router.get(
    "/{requisition_id:int}/email_preview/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["retrieve_requisition"]))
    ],
    response_class=HTMLResponse,
)
async def preview_requisition_email(
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    obj = await utils.retrieve_requisition(session, requisition_id, request_user)
    return await utils.preview_requisition_email(obj, request_user)


@router.post(
    "/{requisition_id:int}/cancel/",
    dependencies=[
        Depends(authorize(requisition_permission_mapper["cancel_requisition"]))
    ],
    response_model=schemas.Requisition,
)
async def cancel_requisition(
    requisition_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.cancel_requisition(session, requisition_id, request_user)
