import importlib.resources as pkg_resources

from fastapi import HTTPException
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit import dateutils
from be_kit.mailing import EmailClient
from be_kit.paginations import PaginationQuery
from be_procurement.transaction.enums import TransactionStatus
from be_uam.user.models import User
from be_uam.organization.models import Group
from ..settings import Settings
from . import models, schemas, repositories


async def create_quotation(
    db: AsyncSession,
    quotation: schemas.QuotationCreate,
    request_user: User,
    _simulate: bool = False,
):
    quotation_data = quotation.model_dump()
    quotation_items = quotation_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.QuotationItemRepository(
        db,
    )
    obj = await repo.acreate(
        **quotation_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=not _simulate,
    )
    for item_data in quotation_items:
        item_data["quotation_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def retrieve_quotation(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_quotations(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.QuotationFilter,
    request_user: User,
    as_options: bool = False,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, as_options)
    return objs


async def update_quotation(
    db: AsyncSession,
    pk: int,
    quotation: schemas.QuotationUpdate,
    request_user: User,
    _simulate: bool = False,
):
    quotation_data = quotation.model_dump()
    quotation_items = quotation_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.QuotationItemRepository(db)
    obj = await repo.aget_or_404(pk)
    if obj.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Only draft quotation or issued RFQ can be updated")
    obj = await repo.aupdate(
        pk,
        **quotation_data,
        status=TransactionStatus.DRAFT,
        last_modified_by_id=request_user.pk,
        _commit=not _simulate,
    )
    await item_repo.adelete_by_quotation_id(pk)
    for item_data in quotation_items:
        item_data["quotation_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def delete_quotation(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def send_quotation_email(
    email: EmailStr,
    quotation: models.Quotation,
    request_user: User,
    settings: Settings,
):
    if quotation.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Quotation is not in draft/issued status")
    group: Group = await request_user.awaitable_attrs.group
    email_client = EmailClient(
        settings=settings.mailing,
        template_dir=pkg_resources.files("symbolix_sales.templates").joinpath("email"),
        test_mode=settings.test_mode,
        debug=settings.debug,
    )
    message = await email_client.get_templated_message(
        template_name="quotation.html",
        subject=f"{group.organization.legal_name} Quotation #{quotation.quotation_id}",
        to=email,
        body={
            "quotation_id": quotation.quotation_id,
            "vendor_name": quotation.vendor.contact.name,
            "organization_name": group.organization.legal_name,
            "items": quotation.items,
            "notes": quotation.notes or "-",
            "year": dateutils.now().year,
        },
    )
    await email_client.send_message(message)


async def preview_quotation_email(
    quotation: models.Quotation,
    request_user: User,
    settings: Settings,
):
    group: Group = await request_user.awaitable_attrs.group
    email_client = EmailClient(
        settings=settings.mailing,
        template_dir=pkg_resources.files("symbolix_sales.templates").joinpath("email"),
        test_mode=settings.test_mode,
        debug=settings.debug,
    )
    preview = await email_client.get_html_preview(
        template_name="quotation.html",
        body={
            "quotation_id": quotation.quotation_id,
            "vendor_name": quotation.vendor.contact.name,
            "organization_name": group.organization.legal_name,
            "items": quotation.items,
            "notes": quotation.notes or "-",
            "year": dateutils.now().year,
        },
    )
    return preview


async def set_quotation_to_issued(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status not in [TransactionStatus.DRAFT, TransactionStatus.ISSUED]:
        raise HTTPException(400, "Quotation is not in draft/issued status")
    if obj.status == TransactionStatus.DRAFT:
        await repo.aupdate(
            pk, status=TransactionStatus.ISSUED, last_modified_by_id=request_user.pk
        )


async def approve_quotation(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aupdate(
        pk, status=TransactionStatus.APPROVED, approved_by_id=request_user.pk
    )
    return obj


async def cancel_quotation(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.QuotationRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status in [TransactionStatus.COMPLETED, TransactionStatus.CANCELLED]:
        raise HTTPException(400, "Cannot cancel completed quotation")
    obj = await repo.aupdate(
        pk, status=TransactionStatus.CANCELLED, last_modified_by_id=request_user.pk
    )
    return obj
