from datetime import date
from sqlalchemy import BigInteger, Date, ForeignKey, String, UniqueConstraint
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import (
    BaseModel,
    SoftDeleteMixin,
)
from be_uam.kit.models import MetadataMixin
from be_uam.organization.models import OrganizationMixin

from be_procurement.transaction.enums import (
    TransactionStatus,
    TransactionPaymentTerm,
    TransactionDeliveryTerm,
)
from be_procurement.transaction.models import (
    TransactionMixin,
    TransactionItemMixin,
)
from be_core_services.country.models import VAT, Currency
from be_inventory.product.models import Product
from ..vendor.models import Vendor


class Quotation(
    TransactionMixin, OrganizationMixin, SoftDeleteMixin, MetadataMixin, BaseModel
):
    __tablename__ = "procurement_quotation"

    quotation_id: Mapped[str] = mapped_column(String(11))
    vendor_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Vendor.pk, ondelete="RESTRICT")
    )
    vendor: Mapped[Vendor] = relationship(
        "Vendor",
        passive_deletes=True,
        lazy="selectin",
    )
    items: Mapped[list["QuotationItem"]] = relationship(
        "QuotationItem",
        back_populates="quotation",
        cascade="all, delete",
        passive_deletes=True,
        lazy="selectin",
    )


class QuotationCounter(OrganizationMixin, BaseModel):
    __tablename__ = "procurement_quotation_counter"
    __table_args__ = (
        UniqueConstraint(
            "organization_id",
            "counter_date",
            name=f"uq_{__tablename__}_organization_id_counter_date",
        ),
    )

    counter_date: Mapped[date] = mapped_column(Date())
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class QuotationItem(TransactionItemMixin, BaseModel):
    __tablename__ = "procurement_quotation_item"
    __table_args__ = (
        UniqueConstraint(
            "quotation_id",
            "product_id",
            name=f"uq_{__tablename__}_quotation_id_product_id",
        ),
    )

    quotation_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Quotation.pk, ondelete="CASCADE")
    )
    quotation: Mapped[Quotation] = relationship(
        "Quotation",
        back_populates="items",
        lazy="selectin",
    )
