from typing import Annotated
from fastapi import APIRouter, BackgroundTasks, Depends, Query, status
from fastapi.responses import HTMLResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from . import schemas, utils, enums
from .permissions import quotation_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["quotations"], prefix="/quotation")


@router.post(
    "/",
    dependencies=[Depends(authorize(quotation_permission_mapper["create_quotation"]))],
    response_model=schemas.Quotation,
)
async def create_quotation(
    quotation: schemas.QuotationCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_quotation(session, quotation, request_user)


@router.post(
    "/simulate/",
    dependencies=[Depends(authorize(quotation_permission_mapper["create_quotation"]))],
    response_model=schemas.Quotation,
)
async def simulate_create_quotation(
    quotation: schemas.QuotationCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_quotation(session, quotation, request_user, _simulate=True)


@router.get(
    "/{quotation_id:int}/",
    dependencies=[Depends(authorize(quotation_permission_mapper["retrieve_quotation"]))],
    response_model=schemas.Quotation,
)
async def retrieve_quotation(
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_quotation(session, quotation_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(quotation_permission_mapper["list_quotation"]))],
    response_model=schemas.PaginatedQuotation,
)
async def list_quotation(
    pagination: PaginationQuery = Depends(),
    filters: schemas.QuotationFilter = Depends(),
    ordering: Annotated[list[enums.QuotationOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_quotations(session, pagination, filters, request_user, ordering)


@router.get(
    "/options/",
    dependencies=[
        Depends(authorize(quotation_permission_mapper["list_quotation_option"]))
    ],
    response_model=schemas.PaginatedQuotationOpt,
)
async def list_quotation_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.QuotationFilter = Depends(),
    ordering: Annotated[list[enums.QuotationOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_quotations(
        session, pagination, filters, request_user, ordering
    )


@router.put(
    "/{quotation_id:int}/",
    dependencies=[Depends(authorize(quotation_permission_mapper["update_quotation"]))],
    response_model=schemas.Quotation,
)
async def update_quotation(
    quotation: schemas.QuotationUpdate,
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_quotation(session, quotation_id, quotation, request_user)


@router.put(
    "/simulate/{quotation_id:int}/",
    dependencies=[Depends(authorize(quotation_permission_mapper["update_quotation"]))],
    response_model=schemas.Quotation,
)
async def simulate_update_quotation(
    quotation: schemas.QuotationUpdate,
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_quotation(
        session, quotation_id, quotation, request_user, _simulate=True
    )


@router.delete(
    "/{quotation_id:int}/",
    dependencies=[Depends(authorize(quotation_permission_mapper["delete_quotation"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_quotation(
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_quotation(session, quotation_id, request_user)


@router.post(
    "/{quotation_id:int}/approve/",
    dependencies=[Depends(authorize(quotation_permission_mapper["approve_quotation"]))],
    response_model=schemas.Quotation,
)
async def approve_quotation(
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_quotation(session, quotation_id, request_user)



@router.post(
    "/{quotation_id:int}/send_email/",
    dependencies=[
        Depends(authorize(quotation_permission_mapper["send_quotation_email"]))
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def send_quotation_email(
    quotation_id: int,
    body: schemas.QuotationEmailSend,
    background_tasks: BackgroundTasks,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    obj = await utils.retrieve_quotation(session, quotation_id, request_user)
    background_tasks.add_task(
        utils.send_quotation_email,
        body.email,
        obj,
        request_user,
    )
    await utils.set_quotation_to_issued(session, quotation_id, request_user)


@router.get(
    "/{quotation_id:int}/email_preview/",
    dependencies=[
        Depends(authorize(quotation_permission_mapper["retrieve_quotation"]))
    ],
    response_class=HTMLResponse,
)
async def preview_quotation_email(
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    obj = await utils.retrieve_quotation(session, quotation_id, request_user)
    return await utils.preview_quotation_email(obj, request_user)



@router.post(
    "/{quotation_id:int}/cancel/",
    dependencies=[Depends(authorize(quotation_permission_mapper["cancel_quotation"]))],
    response_model=schemas.Quotation,
)
async def cancel_quotation(
    quotation_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.cancel_quotation(session, quotation_id, request_user)
