import importlib.resources as pkg_resources

from fastapi import HTTPException
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit import dateutils
from be_kit.formatters.accountings import format_currency
from be_kit.mailing import EmailClient
from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.organization.models import Group

from be_procurement.invoice.enums import InvoiceStatus
from ..order.models import Order
from ..order.repositories import OrderRepository
from ..invoice.repositories import InvoiceRepository
from ..settings import Settings
from . import enums, models, schemas, repositories


async def create_payment(
    db: AsyncSession,
    payment: schemas.PaymentCreate,
    request_user: User,
    _simulate: bool = False,
):
    payment_data = payment.model_dump()
    payment_data.pop("payment_id", None)
    print("payment_data being passed to Payment:", payment_data)  # <-- Add this line

    group = await request_user.awaitable_attrs.group
    order_repo = OrderRepository(db, organization_id=group.organization_id)
    repo = repositories.PaymentRepository(db)
    obj = await repo.acreate(
        **payment_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=not _simulate,
    )
    obj = await repo.aget(obj.pk, disable_organization_filter=True)
    return obj


async def retrieve_payment(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_payments(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.PaymentFilter,
    ordering: list[enums.PaymentOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_payment(
    db: AsyncSession,
    pk: int,
    payment: schemas.PaymentUpdate,
    request_user: User,
    _simulate: bool = False,
):
    payment_data = payment.model_dump()
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aupdate(
        pk, **payment_data, last_modified_by_id=request_user.pk, _commit=not _simulate
    )
    obj = await repo.aget(obj.pk)
    return obj


async def delete_payment(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)

async def change_payment_status(
    db: AsyncSession,
    pk: int,
    status: enums.PaymentStatus,
    request_user: User,
    ):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    obj = await repo.aupdate(
        pk, status=status, last_modified_by_id=request_user.pk
    )
    return obj

async def approve_payment(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status != enums.PaymentStatus.DRAFT:
        raise HTTPException(400, "Only draft payment can be updated")
    obj = await repo.aupdate(
        pk, status=enums.PaymentStatus.PAID, approved_by_id=request_user.pk
    )
    return obj

async def validate_payment_amount(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.PaymentRepository(
        db,
        organization_id=group.organization_id,
    )
    payment_obj = await repo.aget_or_404(pk)

    invoice_id = payment_obj.invoice_id
    amount = payment_obj.amount
    if not invoice_id or amount is None:
        return

    invoice_repo = InvoiceRepository(db, organization_id=group.organization_id)
    invoice = await invoice_repo.aget_or_404(invoice_id)
    invoice_amount = invoice.total_net_after_dp

    if amount > invoice_amount:
        raise HTTPException(400, "Payment amount cannot exceed invoice amount")

    # If the amount is exactly the same, mark invoice as PAID
    if amount == invoice_amount:
        await invoice_repo.aupdate(invoice.pk, status=InvoiceStatus.PAID)
        await repo.aupdate(
            pk, status=enums.PaymentStatus.PAID, approved_by_id=request_user.pk
        )

    return payment_obj
