from datetime import date
from decimal import Decimal
import be_core_services
from sqlalchemy import (
    BigInteger,
    Date,
    Enum,
    ForeignKey,
    Integer,
    Numeric,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import (
    BaseModel,
    SoftDeleteMixin,
)
from be_procurement.invoice.models import Invoice
from be_uam.user.models import User
from be_uam.kit.models import MetadataMixin
from be_procurement.vendor.models import Vendor
from be_uam.organization.models import OrganizationMixin
from be_core_services.country.models import Currency

from ..order.models import Order, OrderItem
from . import enums


class Payment(OrganizationMixin, SoftDeleteMixin, MetadataMixin, BaseModel):
    __tablename__ = "procurement_payment"

    vendor_id: Mapped[int] = mapped_column(BigInteger,
        ForeignKey("procurement_vendor.pk", ondelete="RESTRICT"),
        default=None,
        nullable=True,
    )
    vendor: Mapped["Vendor"] = relationship(
        "Vendor",
        lazy="selectin",
    )
    amount: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    status: Mapped[enums.PaymentStatus] = mapped_column(Enum(enums.PaymentStatus), index=True)
    payment_type: Mapped[enums.PaymentType] = mapped_column(Enum(enums.PaymentType), index=True)
    notes: Mapped[str] = mapped_column(String(255), nullable=True)
    payment_date: Mapped[date] = mapped_column(Date)

    approved_by_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey("uam_user.pk", ondelete="SET NULL"),
        default=None,
        nullable=True,
    )

    approved_by: Mapped["User | None"] = relationship(
        User, foreign_keys=[approved_by_id], lazy="selectin",
    )

    currency_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("core_services_currency.pk", ondelete="RESTRICT"),
    )

    currency: Mapped["Currency"] = relationship(
        "Currency",
        foreign_keys=[currency_id],
        lazy="selectin",
    )

    invoice_id: Mapped[int] = mapped_column(
        BigInteger,
        ForeignKey("procurement_invoice.pk", ondelete="RESTRICT"),
        nullable=True,
    )

    invoice: Mapped["Invoice"] = relationship(
        "Invoice",
        foreign_keys=[invoice_id],
        lazy="selectin",
    )


class PaymentCounter(OrganizationMixin, BaseModel):
    __tablename__ = "procurement_payment_counter"
    __table_args__ = (
        UniqueConstraint(
            "organization_id",
            "counter_date",
            name=f"uq_{__tablename__}_organization_id_counter_date",
        ),
    )

    counter_date: Mapped[date] = mapped_column(Date)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)
