from typing import Annotated
from fastapi import APIRouter, BackgroundTasks, Depends, Query, status
from fastapi.responses import HTMLResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_procurement.transaction.enums import TransactionStatus
from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from . import schemas, utils, enums
from .permissions import payment_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["payments"], prefix="/payment")


@router.post(
    "/",
    dependencies=[Depends(authorize(payment_permission_mapper["create_payment"]))],
    response_model=schemas.Payment,
)
async def create_payment(
    payment: schemas.PaymentCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_payment(session, payment, request_user)


@router.get(
    "/{payment_id:int}/",
    dependencies=[Depends(authorize(payment_permission_mapper["retrieve_payment"]))],
    response_model=schemas.Payment,
)
async def retrieve_payment(
    payment_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_payment(session, payment_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(payment_permission_mapper["list_payment"]))],
    response_model=schemas.PaginatedPayment,
)
async def list_payment(
    pagination: PaginationQuery = Depends(),
    filters: schemas.PaymentFilter = Depends(),
    ordering: Annotated[list[enums.PaymentOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_payments(session, pagination, filters, ordering, request_user)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(payment_permission_mapper["list_payment_option"]))],
    response_model=schemas.PaginatedPaymentOpt,
)
async def list_payment_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.PaymentFilter = Depends(),
    ordering: Annotated[list[enums.PaymentOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_payments(session, pagination, filters, ordering, request_user)



@router.put(
    "/{payment_id:int}/",
    dependencies=[Depends(authorize(payment_permission_mapper["update_payment"]))],
    response_model=schemas.Payment,
)
async def update_payment(
    payment: schemas.PaymentUpdate,
    payment_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
   return await utils.update_payment(session, payment_id, payment, request_user)


@router.delete(
    "/{payment_id:int}/",
    dependencies=[Depends(authorize(payment_permission_mapper["delete_payment"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_payment(
    payment_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_payment(session, payment_id, request_user)


@router.post(
    "/{payment_id:int}/approve/",
    dependencies=[Depends(authorize(payment_permission_mapper["approve_payment"]))],
    response_model=schemas.Payment,
)
async def approve_payment(
    payment_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_payment(session, payment_id, request_user)

@router.post(
    "/{payment_id:int}/change_status/",
    dependencies=[Depends(authorize(payment_permission_mapper["approve_payment"]))],
    response_model=schemas.Payment,
)
async def change_payment_status(
    payment_id: int,
    status: enums.PaymentStatus,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.change_payment_status(session, payment_id, status, request_user)

@router.post(
    "/{payment_id:int}/validate/",
    dependencies=[Depends(authorize(payment_permission_mapper["approve_payment"]))],
    response_model=schemas.Payment,
)
async def validate_payment(
    payment_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.validate_payment_amount(session, payment_id, request_user)
