from fastapi import HTTPException
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_inventory.product.application import create_product_movement
from be_inventory.product.schemas import ProductMovementCreate, ProductMovementItemCreateUpdate
from be_procurement.transaction.enums import TransactionStatus
from be_uam.user.models import User
from ..quotation.repositories import QuotationRepository
from ..quotation.models import Quotation
from . import models, schemas, repositories


async def create_order(
    db: AsyncSession,
    order: schemas.OrderCreate,
    request_user: User,
    _simulate: bool = False,
):
    order_data = order.model_dump()
    order_items = order_data.pop("items")
    group = await request_user.awaitable_attrs.group
    quotation_repo = QuotationRepository(db, organization_id=group.organization_id)
    if order_data["quotation_id"] is not None:
        quotation: Quotation = await quotation_repo.aget_or_404(
            order_data["quotation_id"]
        )
        if quotation.status != TransactionStatus.APPROVED:
            raise HTTPException(400, "Quotation is not approved")
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.OrderItemRepository(
        db,
    )
    obj = await repo.acreate(
        **order_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=not _simulate,
    )
    for item_data in order_items:
        item_data["order_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def retrieve_order(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_orders(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.OrderFilter,
    request_user: User,
    as_options: bool = False,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, as_options)
    return objs


async def update_order(
    db: AsyncSession,
    pk: int,
    order: schemas.OrderUpdate,
    request_user: User,
    _simulate: bool = False,
):
    order_data = order.model_dump()
    order_items = order_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.OrderItemRepository(db)
    obj = await repo.aget_or_404(pk)
    if obj.status != TransactionStatus.DRAFT:
        raise HTTPException(400, "Only draft order can be updated")
    obj = await repo.aupdate(
        pk, **order_data, last_modified_by_id=request_user.pk, _commit=not _simulate
    )
    await item_repo.adelete_by_order_id(pk)
    for item_data in order_items:
        item_data["order_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def delete_order(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def approve_order(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status != TransactionStatus.DRAFT:
        raise HTTPException(400, "Only draft order can be updated")
    obj = await repo.aupdate(
        pk, status=TransactionStatus.APPROVED, approved_by_id=request_user.pk
    )

    # For each item in the order, create a product movement
    movement_id = None
    for idx, item in enumerate(obj.items):
        movement = ProductMovementCreate(
            source_warehouse_id=None,
            destination_warehouse_id=item.warehouse_id,
            movement_type="In",
            note=None,
            items=[
                ProductMovementItemCreateUpdate(
                    product_id=item.product_id,
                    quantity=item.quantity,
                )
            ],
        )
        movement_obj = await create_product_movement(db, movement, request_user)
        if idx == 0:
            movement_id = movement_obj.pk

    if movement_id:
        obj = await repo.aupdate(obj.pk, product_movement_id=movement_id)


    return obj


async def cancel_order(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status in [TransactionStatus.COMPLETED, TransactionStatus.CANCELLED]:
        raise HTTPException(400, "Cannot cancel completed order")
    obj = await repo.aupdate(
        pk, status=TransactionStatus.CANCELLED, last_modified_by_id=request_user.pk
    )
    return obj
