from decimal import Decimal

from pydantic import PositiveInt

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import (
    BaseSchema,
    BaseORMSchema,
)
from be_procurement.transaction.enums import (
    TransactionStatus,
    TransactionPaymentTerm,
    TransactionDeliveryTerm,
)
from be_procurement.transaction.schemas import (
    TransactionCreateMixin,
    TransactionUpdateMixin,
    TransactionItemMixin,
    TransactionMixin,
    TransactionFilterMixin,
    TransactionMinMixin,
)
from be_core_services.country.schemas import CurrencyMin, VATMin
from be_inventory.product.schemas import ProductMin, ProductMovementMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin
from ..quotation.schemas import QuotationMin
from ..vendor.schemas import VendorMin


class OrderIdMixin:
    order_id: str


class VendorWriteMixin:
    vendor_id: PositiveInt


class VendorReadMixin:
    vendor: VendorMin


class OrderCreate(VendorWriteMixin, TransactionCreateMixin, BaseSchema):
    quotation_id: "PositiveInt | None"


class OrderUpdate(VendorWriteMixin, TransactionUpdateMixin, BaseSchema):
    pass


class OrderItem(TransactionItemMixin, BaseORMSchema):
    pass


class Order(
    VendorReadMixin,
    OrderIdMixin,
    TransactionMixin,
    MetadataMixin,
    BaseORMSchema,
):
    quotation: "QuotationMin | None"
    product_movement: ProductMovementMin | None
    items: list[OrderItem]


class OrderList(
    VendorReadMixin,
    OrderIdMixin,
    TransactionMinMixin,
    MetadataMixin,
    BaseORMSchema,
):
    quotation: "QuotationMin | None"


class PaginatedOrder(BasePaginatedResponse):
    items: list[OrderList]


class OrderFilter(TransactionFilterMixin, BaseSchema):
    quotation___quotation_id__icontains: str | None = None
    order_id__icontains: str | None = None
    vendor___contact___name__icontains: str | None = None


class OrderMin(VendorReadMixin, OrderIdMixin, TransactionMinMixin, BaseORMSchema):
    quotation: "QuotationMin | None"


class OrderOpt(OrderMin):
    pass


class PaginatedOrderOpt(BasePaginatedResponse):
    items: list[OrderOpt]


VendorMin.model_rebuild()
ProductMin.model_rebuild()
UserMin.model_rebuild()
CurrencyMin.model_rebuild()
VATMin.model_rebuild()
