from datetime import date
from decimal import Decimal
from sqlalchemy import BigInteger, Date, ForeignKey, Numeric, String, UniqueConstraint
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import (
    BaseModel,
    SoftDeleteMixin,
)
from be_uam.kit.models import MetadataMixin
from be_uam.organization.models import OrganizationMixin

from be_procurement.transaction.enums import (
    TransactionStatus,
    TransactionPaymentTerm,
    TransactionDeliveryTerm,
)
from be_procurement.transaction.models import (
    TransactionMixin,
    TransactionItemMixin,
)
from be_core_services.country.models import VAT, Currency
from be_inventory.product.models import Product, ProductMovement
from ..quotation.models import Quotation
from ..vendor.models import Vendor


class Order(
    TransactionMixin, OrganizationMixin, SoftDeleteMixin, MetadataMixin, BaseModel
):
    __tablename__ = "procurement_order"

    order_id: Mapped[str] = mapped_column(String(11))
    quotation_id: Mapped["int | None"] = mapped_column(
        BigInteger,
        ForeignKey(Quotation.pk, ondelete="RESTRICT"),
        unique=True,
        nullable=True,
    )
    quotation: Mapped["Quotation | None"] = relationship(
        "Quotation",
        passive_deletes=True,
        lazy="selectin",
    )
    vendor_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Vendor.pk, ondelete="RESTRICT")
    )
    vendor: Mapped[Vendor] = relationship(
        "Vendor",
        passive_deletes=True,
        lazy="selectin",
    )
    currency: Mapped[Currency] = relationship(
        "Currency",
        passive_deletes=True,
        lazy="selectin",
    )
    product_movement_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(ProductMovement.pk, ondelete="RESTRICT"), nullable=True
    )
    product_movement: Mapped[ProductMovement] = relationship(
        "ProductMovement",
        lazy="selectin",
    )
    items: Mapped[list["OrderItem"]] = relationship(
        "OrderItem",
        back_populates="order",
        cascade="all, delete",
        passive_deletes=True,
        lazy="selectin",
    )
    total_paid: Mapped[Decimal] = mapped_column(Numeric(20, 4), default=0)


class OrderCounter(OrganizationMixin, BaseModel):
    __tablename__ = "procurement_order_counter"
    __table_args__ = (
        UniqueConstraint(
            "organization_id",
            "counter_date",
            name=f"uq_{__tablename__}_organization_id_counter_date",
        ),
    )

    counter_date: Mapped[date] = mapped_column(Date)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class OrderItem(TransactionItemMixin, BaseModel):
    __tablename__ = "procurement_order_item"
    __table_args__ = (
        UniqueConstraint(
            "order_id",
            "product_id",
            name=f"uq_{__tablename__}_order_id_product_id",
        ),
    )

    order_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Order.pk, ondelete="CASCADE")
    )
    order: Mapped[Order] = relationship(
        "Order",
        back_populates="items",
        lazy="selectin",
    )
    total_paid: Mapped[Decimal] = mapped_column(Numeric(20, 4), default=0)
