from typing import Annotated
from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_procurement.transaction.enums import TransactionStatus
from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from . import schemas, utils, enums
from .permissions import order_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["orders"], prefix="/order")


@router.post(
    "/",
    dependencies=[Depends(authorize(order_permission_mapper["create_order"]))],
    response_model=schemas.Order,
)
async def create_order(
    order: schemas.OrderCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_order(session, order, request_user)



@router.post(
    "/simulate/",
    dependencies=[Depends(authorize(order_permission_mapper["create_order"]))],
    response_model=schemas.Order,
)
async def simulate_create_order(
    order: schemas.OrderCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_order(session, order, request_user, _simulate=True)


@router.get(
    "/{order_id:int}/",
    dependencies=[Depends(authorize(order_permission_mapper["retrieve_order"]))],
    response_model=schemas.Order,
)
async def retrieve_order(
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_order(session, order_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(order_permission_mapper["list_order"]))],
    response_model=schemas.PaginatedOrder,
)
async def list_order(
    pagination: PaginationQuery = Depends(),
    filters: schemas.OrderFilter = Depends(),
    ordering: Annotated[list[enums.OrderOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_orders(session, pagination, filters, request_user, ordering)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(order_permission_mapper["list_order_option"]))],
    response_model=schemas.PaginatedOrderOpt,
)
async def list_order_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.OrderFilter = Depends(),
    ordering: Annotated[list[enums.OrderOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_orders(
        session, pagination, filters, request_user, ordering
    )


@router.put(
    "/{order_id:int}/",
    dependencies=[Depends(authorize(order_permission_mapper["update_order"]))],
    response_model=schemas.Order,
)
async def update_order(
    order: schemas.OrderUpdate,
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_order(session, order_id, order, request_user)


@router.put(
    "/simulate/{order_id:int}/",
    dependencies=[Depends(authorize(order_permission_mapper["update_order"]))],
    response_model=schemas.Order,
)
async def simulate_update_order(
    order: schemas.OrderUpdate,
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_order(
        session, order_id, order, request_user, _simulate=True
    )



@router.delete(
    "/{order_id:int}/",
    dependencies=[Depends(authorize(order_permission_mapper["delete_order"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_order(
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_order(session, order_id, request_user)


@router.post(
    "/{order_id:int}/approve/",
    dependencies=[Depends(authorize(order_permission_mapper["approve_order"]))],
    response_model=schemas.Order,
)
async def approve_order(
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_order(session, order_id, request_user)


@router.post(
    "/{order_id:int}/cancel/",
    dependencies=[Depends(authorize(order_permission_mapper["cancel_order"]))],
    response_model=schemas.Order,
)
async def cancel_order(
    order_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.cancel_order(session, order_id, request_user)
