import importlib.resources as pkg_resources

from fastapi import HTTPException
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession

from be_procurement.transaction.enums import TransactionStatus
from be_kit import dateutils
from be_kit.formatters.accountings import format_currency
from be_kit.mailing import EmailClient
from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.organization.models import Group
from ..order.models import Order
from ..order.repositories import OrderRepository
from ..settings import Settings
from . import enums, models, schemas, repositories


async def create_invoice(
    db: AsyncSession,
    invoice: schemas.InvoiceCreate,
    request_user: User,
    _simulate: bool = False,
):
    invoice_data = invoice.model_dump()
    invoice_items = invoice_data.pop("items")
    group = await request_user.awaitable_attrs.group
    order_repo = OrderRepository(db, organization_id=group.organization_id)
    if invoice_data["order_id"] is not None:
        order = await order_repo.aget_or_404(invoice_data["order_id"])
        if order.status != TransactionStatus.APPROVED:
            raise HTTPException(400, "Order is not approved")
    repo = repositories.InvoiceRepository(
        db,
    )
    item_repo = repositories.InvoiceItemRepository(
        db,
    )
    obj = await repo.acreate(
        **invoice_data,
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
        _commit=not _simulate,
    )
    for item_data in invoice_items:
        item_data["invoice_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk, disable_organization_filter=True)
    return obj


async def retrieve_invoice(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_invoices(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.InvoiceFilter,
    request_user: User,
    as_options: bool = False,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, as_options)
    return objs


async def update_invoice(
    db: AsyncSession,
    pk: int,
    invoice: schemas.InvoiceUpdate,
    request_user: User,
    _simulate: bool = False,
):
    invoice_data = invoice.model_dump()
    invoice_items = invoice_data.pop("items")
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(
        db,
        organization_id=group.organization_id,
    )
    item_repo = repositories.InvoiceItemRepository(
        db,
    )
    obj = await repo.aupdate(
        pk, **invoice_data, last_modified_by_id=request_user.pk, _commit=not _simulate
    )
    # TODO: Change to update per item ID
    await item_repo.adelete_by_invoice_id(pk)
    for item_data in invoice_items:
        item_data["invoice_id"] = obj.pk
        await item_repo.acreate(**item_data, _commit=not _simulate)
    db.expire(obj, ["items"])
    obj = await repo.aget(obj.pk)
    return obj


async def delete_invoice(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def approve_invoice(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status != enums.InvoiceStatus.DRAFT:
        raise HTTPException(400, "Only draft invoice can be updated")
    obj = await repo.aupdate(
        pk, status=enums.InvoiceStatus.APPROVED, approved_by_id=request_user.pk
    )
    return obj


async def cancel_invoice(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.InvoiceRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status in [
        enums.InvoiceStatus.PAID,
        enums.InvoiceStatus.PARTIALLY_PAID,
        enums.InvoiceStatus.OVER_PAID,
        enums.InvoiceStatus.CANCELLED,
    ]:
        raise HTTPException(400, "Cannot cancel completed invoice")
    obj = await repo.aupdate(
        pk, status=enums.InvoiceStatus.CANCELLED, last_modified_by_id=request_user.pk
    )
    return obj
