from datetime import date
from sqlalchemy import Select, select, delete, func

from be_kit.repositories import BaseRepository

from . import models


class InvoiceRepository(BaseRepository):
    model = models.Invoice
    option_fields = [
        "pk",
        "invoice_id",
        "status",
    ]
    filter_by_organization = True

    def add_organization_filter(self, query: Select):
        query = query.filter_by(organization_id=self.organization_id)
        return query

    async def acreate(self, _commit: bool = True, **kwargs):
        today = date.today()
        counter = await self.aget_counter(today, kwargs["organization_id"])
        kwargs["invoice_id"] = self.get_invoice_id(today, counter)
        if not _commit:
            max_pk = await self.session.execute(select(func.max(self.model.pk)))
            max_pk = max_pk.scalar() or 0
            kwargs["pk"] = max_pk + 1
        return await super().acreate(_commit, **kwargs)

    async def aget_counter(self, dt: date, organization_id: int):
        query = select(models.InvoiceCounter).filter_by(
            organization_id=organization_id, counter_date=dt
        )
        obj = await self.session.execute(query)
        obj = obj.scalar()
        if obj is None:
            obj = models.InvoiceCounter(
                organization_id=organization_id, counter_date=dt, counter=0
            )
            self.session.add(obj)
        else:
            obj.counter += 1
            self.session.add(obj)
        await self.session.flush()
        return obj.counter

    def get_invoice_id(self, dt: date, counter: int):
        return f"PI{dt.strftime('%y%m%d')}{counter:03d}"


class InvoiceItemRepository(BaseRepository):
    model = models.InvoiceItem

    async def acreate(self, _commit: bool = True, **kwargs):
        if not _commit:
            max_pk = await self.session.execute(select(func.max(self.model.pk)))
            max_pk = max_pk.scalar() or 0
            kwargs["pk"] = max_pk + 1
        return await super().acreate(_commit, **kwargs)

    # TODO: Change to update per item ID
    async def adelete_by_invoice_id(self, invoice_id: int):
        query = delete(self.model).filter_by(invoice_id=invoice_id)
        await self.session.execute(query)
