from datetime import date
from decimal import Decimal
from sqlalchemy import (
    BigInteger,
    Date,
    Enum,
    ForeignKey,
    Integer,
    Numeric,
    String,
    UniqueConstraint,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import (
    BaseModel,
    SoftDeleteMixin,
)
from be_procurement.transaction.enums import (
    TransactionStatus,
    TransactionPaymentTerm,
    TransactionDeliveryTerm,
)
from be_procurement.transaction.models import CalculationMixin
from be_uam.user.models import User
from be_uam.kit.models import MetadataMixin
from be_uam.organization.models import OrganizationMixin

from ..order.models import Order, OrderItem
from . import enums


class Invoice(OrganizationMixin, SoftDeleteMixin, CalculationMixin, MetadataMixin, BaseModel):
    __tablename__ = "procurement_invoice"

    status: Mapped["enums.InvoiceStatus"] = mapped_column(
        Enum(enums.InvoiceStatus),
        index=True,
        default=enums.InvoiceStatus.DRAFT,
    )
    invoice_id: Mapped[str] = mapped_column(String(11))
    order_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Order.pk, ondelete="RESTRICT")
    )
    order: Mapped[Order] = relationship(
        "Order",
        lazy="selectin",
    )
    payment_term: Mapped["TransactionPaymentTerm"] = mapped_column(
        Enum(TransactionPaymentTerm), index=True, nullable=True
    )
    payment_n: Mapped["int | None"] = mapped_column(Integer, nullable=True)
    payment_dp_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    payment_dp: Mapped["Decimal | None"] = mapped_column(Numeric(20, 4), nullable=True)
    delivery_term: Mapped["TransactionDeliveryTerm | None"] = mapped_column(
        Enum(TransactionDeliveryTerm), index=True, nullable=True
    )
    discount_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    discount_amount: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_global_discount: Mapped[Decimal] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    total_before_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net_after_dp: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_paid: Mapped[Decimal] = mapped_column(Numeric(20, 4), default=0)
    notes: Mapped[str] = mapped_column(String(255), nullable=True)
    approved_by_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey("uam_user.pk", ondelete="SET NULL"),
        default=None,
        nullable=True,
    )

    approved_by: Mapped["User | None"] = relationship(
        User, foreign_keys=[approved_by_id], lazy="selectin"
    )
    items: Mapped[list["InvoiceItem"]] = relationship(
        "InvoiceItem",
        back_populates="invoice",
        cascade="all, delete",
        passive_deletes=True,
        lazy="selectin",
    )


class InvoiceCounter(OrganizationMixin, BaseModel):
    __tablename__ = "procurement_invoice_counter"
    __table_args__ = (
        UniqueConstraint(
            "organization_id",
            "counter_date",
            name=f"uq_{__tablename__}_organization_id_counter_date",
        ),
    )

    counter_date: Mapped[date] = mapped_column(Date)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)


class InvoiceItem(BaseModel):
    __tablename__ = "procurement_invoice_item"

    invoice_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(Invoice.pk, ondelete="CASCADE")
    )
    invoice: Mapped[Invoice] = relationship(
        "Invoice",
        back_populates="items",
        lazy="selectin",
    )
    order_item_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(OrderItem.pk, ondelete="RESTRICT")
    )
    order_item: Mapped[OrderItem] = relationship(OrderItem)
    quantity: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    price: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    discount_rate: Mapped["Decimal | None"] = mapped_column(
        Numeric(5, 4), nullable=True
    )
    discount_amount: Mapped["Decimal | None"] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    vat_rate: Mapped[Decimal] = mapped_column(Numeric(5, 4))
    total: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_discount: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_global_discount: Mapped[Decimal] = mapped_column(
        Numeric(20, 4), nullable=True
    )
    total_before_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4), nullable=True)
    total_vat: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_net: Mapped[Decimal] = mapped_column(Numeric(20, 4))
    total_paid: Mapped[Decimal] = mapped_column(Numeric(20, 4), default=0)
