from typing import Annotated
from fastapi import APIRouter, BackgroundTasks, Depends, status
from fastapi.params import Query
from fastapi.responses import HTMLResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_procurement.transaction.enums import TransactionStatus
from be_kit.paginations import PaginationQuery
from be_uam.auth.utils import get_request_user, authorize
from be_uam.user.models import User
from . import schemas, utils, enums
from .permissions import invoice_permission_mapper
from ..databases import get_async_session
from ..settings import Settings


router = APIRouter(tags=["invoices"], prefix="/invoice")


@router.post(
    "/",
    dependencies=[Depends(authorize(invoice_permission_mapper["create_invoice"]))],
    response_model=schemas.Invoice,
)
async def create_invoice(
    invoice: schemas.InvoiceCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_invoice(session, invoice, request_user)


@router.post(
    "/simulate/",
    dependencies=[Depends(authorize(invoice_permission_mapper["create_invoice"]))],
    response_model=schemas.Invoice,
)
async def simulate_create_invoice(
    invoice: schemas.InvoiceCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_invoice(session, invoice, request_user, _simulate=True)


@router.get(
    "/{invoice_id:int}/",
    dependencies=[Depends(authorize(invoice_permission_mapper["retrieve_invoice"]))],
    response_model=schemas.Invoice,
)
async def retrieve_invoice(
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_invoice(session, invoice_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(invoice_permission_mapper["list_invoice"]))],
    response_model=schemas.PaginatedInvoice,
)
async def list_invoice(
    pagination: PaginationQuery = Depends(),
    filters: schemas.InvoiceFilter = Depends(),
    ordering: Annotated[list[enums.InvoiceOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_invoices(session, pagination, filters, request_user, ordering)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(invoice_permission_mapper["list_invoice_option"]))],
    response_model=schemas.PaginatedInvoiceOpt,
)
async def list_invoice_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.InvoiceFilter = Depends(),
    ordering: Annotated[list[enums.InvoiceOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_invoices(
        session, pagination, filters, request_user, ordering
    )


@router.put(
    "/{invoice_id:int}/",
    dependencies=[Depends(authorize(invoice_permission_mapper["update_invoice"]))],
    response_model=schemas.Invoice,
)
async def update_invoice(
    invoice: schemas.InvoiceUpdate,
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_invoice(session, invoice_id, invoice, request_user)


@router.put(
    "/simulate/{invoice_id:int}/",
    dependencies=[Depends(authorize(invoice_permission_mapper["update_invoice"]))],
    response_model=schemas.Invoice,
)
async def simulate_update_invoice(
    invoice: schemas.InvoiceUpdate,
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_invoice(
        session, invoice_id, invoice, request_user, _simulate=True
    )


@router.delete(
    "/{invoice_id:int}/",
    dependencies=[Depends(authorize(invoice_permission_mapper["delete_invoice"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_invoice(
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_invoice(session, invoice_id, request_user)


@router.post(
    "/{invoice_id:int}/approve/",
    dependencies=[Depends(authorize(invoice_permission_mapper["approve_invoice"]))],
    response_model=schemas.Invoice,
)
async def approve_invoice(
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_invoice(session, invoice_id, request_user)

@router.post(
    "/{invoice_id:int}/cancel/",
    dependencies=[Depends(authorize(invoice_permission_mapper["cancel_invoice"]))],
    response_model=schemas.Invoice,
)
async def cancel_invoice(
    invoice_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.cancel_invoice(session, invoice_id, request_user)
