from datetime import datetime, timezone
import io

import pandas as pd
from sqlalchemy.ext.asyncio import AsyncSession
from fastapi import HTTPException

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from . import enums, schemas, repositories
from be_core_services.country.repositories import CurrencyRepository
from be_core_services.country.schemas import CurrencyFilter
from be_expense.expense.enums import ExpenseStatus
from be_uam.auth.utils import check_permission
# ExpenseCategory


async def create_expense_category(
    db: AsyncSession,
    expense_category: schemas.ExpenseCategoryCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseCategoryRepository(db)
    obj = await repo.acreate(
        **expense_category.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_expense_category(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseCategoryRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def update_expense_category(
    db: AsyncSession,
    pk: int,
    expense_category: schemas.ExpenseCategoryUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseCategoryRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aupdate(pk, **expense_category.model_dump())
    return obj


async def list_expense_categories(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.ExpenseCategoryFilter,
    ordering: list[enums.ExpenseCategoryOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseCategoryRepository(
        db, organization_id=group.organization_id
    )
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_expense_category(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseCategoryRepository(
        db, organization_id=group.organization_id
    )
    await repo.adelete(pk)


# Expense


async def create_expense(
    db: AsyncSession,
    expense: schemas.ExpenseCreate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    obj = await repo.acreate(
        **expense.model_dump(),
        organization_id=group.organization_id,
        created_by_id=request_user.pk,
    )
    return obj


async def retrieve_expense(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    return obj


async def update_expense(
    db: AsyncSession,
    pk: int,
    expense: schemas.ExpenseUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    if obj.status != ExpenseStatus.DRAFT:
        raise HTTPException(400, "Only draft order can be updated")
    obj = await repo.aupdate(pk, **expense.model_dump())
    return obj


async def list_expenses(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.ExpenseFilter,
    ordering: list[enums.ExpenseOrdering] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def delete_expense(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    await repo.adelete(pk)


async def approve_expense(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status != ExpenseStatus.DRAFT:
        raise HTTPException(400, "Only draft order can be updated")
    obj = await repo.aupdate(
        pk, status=ExpenseStatus.APPROVED, approved_by_id=request_user.pk
    )
    return obj

async def cancel_expense(
    db: AsyncSession,
    pk: int,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrderRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if obj.status in [ExpenseStatus.COMPLETED, ExpenseStatus.CANCELLED]:
        raise HTTPException(400, "Cannot cancel completed order")
    obj = await repo.aupdate(
        pk, status=ExpenseStatus.CANCELLED, last_modified_by_id=request_user.pk
    )
    return obj

async def set_expense_status(
    db: AsyncSession,
    pk: int,
    status_update: schemas.ApprovalStatusUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(
        db,
        organization_id=group.organization_id,
    )

    obj = await repo.aget_or_404(pk)
    current_status = obj.status
    new_status = status_update.status

    # Enforce allowed transitions
    if current_status == ExpenseStatus.DRAFT and new_status != ExpenseStatus.SUBMITTED:
        raise HTTPException(400, "Draft can only be set to Submitted")
    if current_status == ExpenseStatus.SUBMITTED and new_status not in [ExpenseStatus.APPROVED, ExpenseStatus.CANCELLED]:
        raise HTTPException(400, "Submitted can only be set to Approved or Canceled")
    if current_status == ExpenseStatus.APPROVED and new_status != ExpenseStatus.COMPLETED:
        raise HTTPException(400, "Approved can only be set to Completed")
    if current_status not in [ExpenseStatus.DRAFT, ExpenseStatus.SUBMITTED, ExpenseStatus.APPROVED]:
        raise HTTPException(400, "Status change not allowed from current state")

    if status_update.status == ExpenseStatus.APPROVED:
        obj = await repo.aupdate(pk, **status_update.model_dump(), approved_by_id=request_user.pk)
    if status_update.status == ExpenseStatus.CANCELLED:
        obj = await repo.aupdate(pk, **status_update.model_dump(), cancelled_by_id=request_user.pk)

    obj = await repo.aupdate(pk, **status_update.model_dump())
    return obj


async def handle_expense_download(
    db: AsyncSession,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
    expenses = await repo.arecords()

    account_rows = []
    for expense in expenses:
        account_rows.append(
            {
                "description": expense.description,
                "amount": expense.amount,
                "expense_date": expense.expense_date,
                "category": expense.category.name if expense.category else None,
                "status": expense.status.value,
                "employee": expense.employee.username if expense.employee else None,
            }
        )

    df = pd.DataFrame(account_rows)
    output = io.BytesIO()
    with pd.ExcelWriter(output, engine="openpyxl") as writer:
        df.to_excel(writer, sheet_name="account", index=False)
    output.seek(0)
    return output

# async def handle_expense_upload(
#     db: AsyncSession,
#     expense_upload: schemas.ExpenseUpload,
#     request_user: User,
# ):
#     group = await request_user.awaitable_attrs.group
#     repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
#     category_repo = repositories.ExpenseCategoryRepository(db, organization_id=group.organization_id)
#     currency_repo = CurrencyRepository(db)  # Add this repository

#     content = await expense_upload.file.read()
#     df = pd.read_excel(io.BytesIO(content))

#     required_columns = {"name", "category", "description", "expense_type", "expense_unit", "price", "currency_iso_code"}
#     if not required_columns.issubset(df.columns):
#         missing = required_columns - set(df.columns)
#         raise ValueError(f"Missing required columns: {', '.join(missing)}")

#     created_expenses = []
#     for _, row in df.iterrows():
#         # Validate row using ExpenseValidator schema
#         validated = schemas.ExpenseValidator.model_validate({
#             "name": row["name"],
#             "category": row["category"],
#             "description": row["description"],
#             "expense_type": row["expense_type"],
#             "expense_unit": row["expense_unit"],
#             "price": row["price"],
#             "currency_iso_code": row["currency_iso_code"],
#         })

#         # Get category pk from category name
#         category_obj = await category_repo.arecords(
#             schemas.ExpenseCategoryFilter(name=validated.category),
#             PaginationQuery(limit=1, offset=0),
#             None
#         )
#         items = list(category_obj["items"])
#         if not items or len(items) == 0:
#             raise ValueError(f"Category '{validated.category}' not found")
#         category_pk = items[0].pk

#         # Get currency pk from iso code
#         currency_obj = await currency_repo.arecords(
#             CurrencyFilter(iso=validated.currency_iso_code),
#             PaginationQuery(limit=1, offset=0),
#             None
#         )
#         currency_items = list(currency_obj["items"])
#         if not currency_items or len(currency_items) == 0:
#             raise ValueError(f"Currency '{validated.currency_iso_code}' not found")
#         currency_pk = currency_items[0].pk

#         expense_data = {
#             "name": validated.name,
#             "description": validated.description,
#             "category_id": category_pk,
#             "expense_type": validated.expense_type,
#             "expense_unit": validated.expense_unit,
#             "price": validated.price,
#             "currency_id": currency_pk,
#         }
#         obj = await repo.acreate(
#             **expense_data,
#             organization_id=group.organization_id,
#             created_by_id=request_user.pk,
#         )
#         created_expenses.append(schemas.ExpenseMin.model_validate(obj).model_dump())

#     return created_expenses


# async def handle_expense_download(
#     db: AsyncSession,
#     request_user: User,
# ):
#     group = await request_user.awaitable_attrs.group
#     repo = repositories.ExpenseRepository(db, organization_id=group.organization_id)
#     expenses = await repo.arecords()

#     expense_rows = []
#     for expense in expenses:
#         expense_rows.append(
#             {
#                 "name": expense.name,
#                 "category": expense.category.name,
#                 "description": expense.description,
#                 "expense_type": expense.expense_type,
#                 "expense_unit": expense.expense_unit,
#                 "price": expense.price,
#                 "currency_iso_code": expense.currency.iso,
#             }
#         )

#     df = pd.DataFrame(expense_rows)
#     output = io.BytesIO()
#     with pd.ExcelWriter(output, engine="openpyxl") as writer:
#         df.to_excel(writer, sheet_name="expense", index=False)
#     output.seek(0)
#     return output

# Currency Utils

async def get_all_currency_iso_codes(db, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = CurrencyRepository(db, organization_id=group.organization_id)
    currencies = await repo.arecords()
    # Assuming each currency has an 'iso' attribute
    return [cur.iso for cur in currencies]

async def get_all_expense_category_names(db: AsyncSession, request_user: User):
    group = await request_user.awaitable_attrs.group
    from .repositories import ExpenseCategoryRepository
    repo = ExpenseCategoryRepository(db, organization_id=group.organization_id)
    categories = await repo.arecords()
    if hasattr(categories, "all"):
        categories = categories.all()
    return [cat.name for cat in categories]
