from datetime import date, datetime
from fastapi import File, UploadFile
from typing_extensions import Annotated
from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema, NominalType
from be_core_services.country.schemas import CurrencyMin
from be_uam.kit.schemas import MetadataMixin
from be_uam.user.schemas import UserMin

from be_accounting.coa.schemas import AccountMin
from . import enums


# ExpenseCategory


class BaseExpenseCategory:
    name: str
    description: str | None

class ExpenseCategoryWriteMixin:
    account_id: PositiveInt


class ExpenseCategoryCreate(BaseExpenseCategory, ExpenseCategoryWriteMixin, BaseSchema):
    pass


class ExpenseCategoryUpdate(BaseExpenseCategory, ExpenseCategoryWriteMixin, BaseSchema):
    pass


class ExpenseCategory(MetadataMixin, BaseExpenseCategory, BaseORMSchema):
    account: AccountMin


class PaginatedExpenseCategory(BasePaginatedResponse):
    items: list[ExpenseCategory]


class ExpenseCategoryFilter(BaseSchema):
    name: str | None = None
    description: str | None = None
    account_id: PositiveInt | None = None


class ExpenseCategoryMin(BaseExpenseCategory, BaseORMSchema):
   account: AccountMin


class ExpenseCategoryOpt(ExpenseCategoryMin):
    pass


class PaginatedExpenseCategoryOpt(BasePaginatedResponse):
    items: list[ExpenseCategoryOpt]


# Expense


class BaseExpense:
    description: str | None
    expense_date: date | None
    notes: str | None
    amount: NominalType | None


class ExpenseIdentifierMixin:
    identifier: Annotated[
        str,
        StringConstraints(min_length=1, max_length=10),
    ]

class ExpenseWriteMixin:
    category_id: PositiveInt
    currency_id: PositiveInt
    employee_id: PositiveInt


class ExpenseCreate(BaseExpense, ExpenseWriteMixin, BaseSchema):
    pass


class ExpenseUpdate(BaseExpense, ExpenseWriteMixin, BaseSchema):
    pass


class Expense(MetadataMixin, ExpenseIdentifierMixin, BaseExpense, BaseORMSchema):
    status: enums.ExpenseStatus
    employee: UserMin
    category: ExpenseCategoryMin
    currency: CurrencyMin


class PaginatedExpense(BasePaginatedResponse):
    items: list[Expense]


class ExpenseFilter(BaseSchema):
    description: str | None = None
    identifier: str | None = None
    employee_id: PositiveInt | None = None
    category_id: PositiveInt | None = None
    currency_id: PositiveInt | None = None


class ExpenseMin(BaseExpense, ExpenseIdentifierMixin, BaseORMSchema):
    status: enums.ExpenseStatus
    category: ExpenseCategoryMin
    currency: CurrencyMin
    employee: UserMin

class ExpenseOpt(ExpenseMin):
    pass


class PaginatedExpenseOpt(BasePaginatedResponse):
    items: list[ExpenseOpt]


class ApprovalStatusMixin:
    status: enums.ExpenseStatus

class ApprovalStatusUpdate(ApprovalStatusMixin, BaseSchema):
    pass


UserMin.model_rebuild()
