from datetime import date, datetime
from decimal import Decimal

from sqlalchemy import (
    BigInteger,
    DateTime,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
    Date,
)
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit.databases import BaseModel, SoftDeleteMixin, NominalField
from be_uam.user.models import User
from be_uam.kit.models import MetadataMixin, OrganizationMixin
from be_core_services.country.models import Currency
from be_accounting.coa.models import Account
from be_expense.expense.mixin import ApprovalMetadataMixin
from . import enums


class ExpenseCategory(SoftDeleteMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "expense_category"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    name: Mapped[str] = mapped_column(String(100), index=True)
    description: Mapped[str | None] = mapped_column(String(500), nullable=True)

    account_id: Mapped[int] = mapped_column(
        ForeignKey("accounting_coa_account.pk", ondelete="RESTRICT"),
    )
    account: Mapped[Account] = relationship(
        "Account",
        passive_deletes=True,
        lazy="selectin",
    )


class Expense(SoftDeleteMixin, ApprovalMetadataMixin, OrganizationMixin, MetadataMixin, BaseModel):
    __tablename__ = "expense"
    __table_args__ = (UniqueConstraint("organization_id", "identifier"),)

    description: Mapped[str | None] = mapped_column(String(500), nullable=True)
    identifier: Mapped[str] = mapped_column(String(10), unique=True, index=True)
    expense_date: Mapped[date] = mapped_column(Date, index=True)
    notes: Mapped[str | None] = mapped_column(String(2000), nullable=True)
    status: Mapped[enums.ExpenseStatus] = mapped_column(
        Enum(enums.ExpenseStatus), nullable=False, default=enums.ExpenseStatus.DRAFT
    )

    category_id: Mapped[int] = mapped_column(
        ForeignKey(f"{ExpenseCategory.__tablename__}.pk", ondelete="RESTRICT"),
    )
    category: Mapped[ExpenseCategory] = relationship(
        "ExpenseCategory",
        passive_deletes=True,
        lazy="selectin",
    )

    currency_id: Mapped[int] = mapped_column(
        ForeignKey(f"{Currency.__tablename__}.pk", ondelete="RESTRICT"),
    )
    currency: Mapped[Currency] = relationship(
        "Currency",
        passive_deletes=True,
        lazy="selectin",
    )

    employee_id: Mapped[int] = mapped_column(
        ForeignKey("uam_user.pk", ondelete="CASCADE"),
    )
    employee: Mapped[User] = relationship(
        "User",
        foreign_keys=[employee_id],
        passive_deletes=True,
        lazy="selectin",
    )

    amount: Mapped[Decimal] = mapped_column(NominalField, default=Decimal("0.00"))


class ExpenseCounter(OrganizationMixin, BaseModel):
    __tablename__ = "expense_counter"
    __table_args__ = (UniqueConstraint("organization_id", "counter_date"),)

    counter_date: Mapped[date] = mapped_column(Date, index=True)
    counter: Mapped[int] = mapped_column(BigInteger, default=0)
