"""
Mixins for database models.
"""

from datetime import datetime
from typing import Optional, List, Any, ClassVar, TYPE_CHECKING, ForwardRef

from sqlalchemy import (
    func,
    ForeignKey,
    Identity,
    DateTime,
    String,
    Text,
    BigInteger,
    Integer,
)
from sqlalchemy.orm import relationship, Mapped, mapped_column, declared_attr
from be_uam.user.models import User


class ApprovalMetadataMixin:
    """
    Mixin that adds metadata fields to a model:
    - created_at
    - last_modified_at
    - created_by_id (with relationship to User)
    - last_modified_by_id (with relationship to User)
    """

    @declared_attr
    @classmethod
    def approved_at(cls) -> Mapped[datetime]:
        return mapped_column(DateTime(timezone=True), server_default=func.now())


    @declared_attr
    @classmethod
    def approved_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def approved_by(cls) -> Mapped["User | None"]:
        return relationship(
            User,
            foreign_keys=[cls.approved_by_id],
            passive_deletes=True,
            lazy="selectin",
        )

    @declared_attr
    @classmethod
    def rejected_at(cls) -> Mapped[datetime]:
        return mapped_column(DateTime(timezone=True), nullable=True)

    @declared_attr
    @classmethod
    def cancelled_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def cancelled_by(cls) -> Mapped["User | None"]:
        return relationship(
            User,
            foreign_keys=[cls.cancelled_by_id],
            passive_deletes=True,
            lazy="selectin",
        )
