import io
import pandas as pd
from typing import Annotated

from openpyxl import load_workbook
from openpyxl.worksheet.datavalidation import DataValidation

from be_kit.exim.utils import get_version_upload_template
from fastapi import APIRouter, Depends, Query, status
from fastapi.params import Form
from fastapi.responses import StreamingResponse
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from be_uam.user.models import User
from be_uam.auth.utils import get_request_user, authorize
from ..databases import get_async_session
from . import enums, schemas, utils
from .permissions import (
    expense_category_permission_mapper,
    expense_permission_mapper,
)


# Expense Category


expense_category_router = APIRouter(
    tags=["expense_categories"], prefix="/expense_category"
)


@expense_category_router.post(
    "/",
    dependencies=[
        Depends(
            authorize(expense_category_permission_mapper["create_expense_category"])
        )
    ],
    response_model=schemas.ExpenseCategory,
    status_code=status.HTTP_201_CREATED,
)
async def create_expense_category(
    expense_category: schemas.ExpenseCategoryCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_expense_category(session, expense_category, request_user)


@expense_category_router.get(
    "/{expense_category_id:int}/",
    dependencies=[
        Depends(
            authorize(expense_category_permission_mapper["retrieve_expense_category"])
        )
    ],
    response_model=schemas.ExpenseCategory,
)
async def retrieve_expense_category(
    expense_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_expense_category(
        session, expense_category_id, request_user
    )


@expense_category_router.get(
    "/",
    dependencies=[
        Depends(authorize(expense_category_permission_mapper["list_expense_category"]))
    ],
    response_model=schemas.PaginatedExpenseCategory,
)
async def list_expense_category(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ExpenseCategoryFilter = Depends(),
    ordering: Annotated[list[enums.ExpenseCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_expense_categories(
        session, pagination, filters, ordering, request_user
    )


@expense_category_router.get(
    "/options/",
    dependencies=[
        Depends(
            authorize(
                expense_category_permission_mapper["list_expense_category_option"]
            )
        )
    ],
    response_model=schemas.PaginatedExpenseCategoryOpt,
)
async def list_expense_category_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ExpenseCategoryFilter = Depends(),
    ordering: Annotated[list[enums.ExpenseCategoryOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_expense_categories(
        session, pagination, filters, ordering, request_user
    )


@expense_category_router.put(
    "/{expense_category_id:int}/",
    dependencies=[
        Depends(
            authorize(expense_category_permission_mapper["update_expense_category"])
        )
    ],
    response_model=schemas.ExpenseCategory,
)
async def update_expense_category(
    expense_category_id: int,
    expense_category: schemas.ExpenseCategoryUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_expense_category(
        session, expense_category_id, expense_category, request_user
    )


@expense_category_router.delete(
    "/{expense_category_id:int}/",
    dependencies=[
        Depends(
            authorize(expense_category_permission_mapper["delete_expense_category"])
        )
    ],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_expense_category(
    expense_category_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_expense_category(session, expense_category_id, request_user)


# Expense


expense_router = APIRouter(tags=["expenses"], prefix="/expense")


@expense_router.post(
    "/",
    dependencies=[Depends(authorize(expense_permission_mapper["create_expense"]))],
    response_model=schemas.Expense,
    status_code=status.HTTP_201_CREATED,
)
async def create_expense(
    expense: schemas.ExpenseCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_expense(session, expense, request_user)


@expense_router.get(
    "/{expense_id:int}/",
    dependencies=[Depends(authorize(expense_permission_mapper["retrieve_expense"]))],
    response_model=schemas.Expense,
)
async def retrieve_expense(
    expense_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_expense(session, expense_id, request_user)


@expense_router.get(
    "/",
    dependencies=[Depends(authorize(expense_permission_mapper["list_expense"]))],
    response_model=schemas.PaginatedExpense,
)
async def list_expense(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ExpenseFilter = Depends(),
    ordering: Annotated[list[enums.ExpenseOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_expenses(
        session, pagination, filters, ordering, request_user
    )


@expense_router.get(
    "/options/",
    dependencies=[Depends(authorize(expense_permission_mapper["list_expense_option"]))],
    response_model=schemas.PaginatedExpenseOpt,
)
async def list_expense_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.ExpenseFilter = Depends(),
    ordering: Annotated[list[enums.ExpenseOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_expenses(
        session, pagination, filters, ordering, request_user
    )


@expense_router.put(
    "/{expense_id:int}/",
    dependencies=[Depends(authorize(expense_permission_mapper["update_expense"]))],
    response_model=schemas.Expense,
)
async def update_expense(
    expense_id: int,
    expense: schemas.ExpenseUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_expense(session, expense_id, expense, request_user)


@expense_router.delete(
    "/{expense_id:int}/",
    dependencies=[Depends(authorize(expense_permission_mapper["delete_expense"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_expense(
    expense_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_expense(session, expense_id, request_user)


@expense_router.post(
    "/{expense_id:int}/approve/",
    dependencies=[Depends(authorize(expense_permission_mapper["approve_expense"]))],
    response_model=schemas.Expense,
)
async def approve_expense(
    expense_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.approve_expense(session, expense_id, request_user)


@expense_router.post(
    "/{expense_id:int}/cancel/",
    dependencies=[Depends(authorize(expense_permission_mapper["cancel_expense"]))],
    response_model=schemas.Expense,
)
async def cancel_expense(
    expense_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.cancel_expense(session, expense_id, request_user)

@expense_router.put(
    "/{expense_id:int}/set_status/",
    dependencies=[Depends(authorize(expense_permission_mapper["approve_expense"]))],
    response_model=schemas.Expense
)
async def set_expense_status(
    expense_id: int,
    status_update: schemas.ApprovalStatusUpdate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.set_expense_status(session, expense_id, status_update, request_user)


@expense_router.get(
    "/download/",
    dependencies=[Depends(authorize(expense_permission_mapper["download_expense"]))],
)
async def download_expense(
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    buffer = await utils.handle_expense_download(session, request_user)
    return StreamingResponse(
        buffer,
        media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
        headers={"Content-Disposition": 'attachment; filename="Expenses.xlsx"'},
    )


# @expense_router.get(
#     "/upload/template/",
#     dependencies=[
#         Depends(authorize(expense_permission_mapper["download_upload_expense_template"]))
#     ],
# )
# async def download_upload_expense_template(
#     request_user: User = Depends(get_request_user),
#     session: AsyncSession = Depends(get_async_session)
# ):
#     expense_template = await get_version_upload_template(schemas.ExpenseValidator)

#     # Fetch categories and currencies from DB
#     categories = await utils.get_all_expense_category_names(session, request_user)
#     currencies = await utils.get_all_currency_iso_codes(session, request_user)

#     buffer = io.BytesIO()
#     with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
#         expense_template.to_excel(writer, sheet_name="Expense", index=False)
#     buffer.seek(0)

#     wb = load_workbook(buffer)
#     ws = wb["Expense"]

#     # Add a hidden sheet for dropdown sources
#     ws_lists = wb.create_sheet("lists")
#     for i, cat in enumerate(categories, 1):
#         ws_lists[f"A{i}"] = cat
#     for i, iso in enumerate(currencies, 1):
#         ws_lists[f"B{i}"] = iso
#     wb.active = 0  # Set back to main sheet

#     # Import your enums
#     from .enums import ExpenseType, ExpenseUnit

#     # Get enum values
#     expense_type_values = [e.value for e in ExpenseType]
#     expense_unit_values = [e.value for e in ExpenseUnit]

#     # Find columns
#     header = [cell.value for cell in ws[1]]
#     from openpyxl.utils import get_column_letter
#     category_col = get_column_letter(header.index("category") + 1)
#     currency_col = get_column_letter(header.index("currency_iso_code") + 1)
#     expense_type_col = get_column_letter(header.index("expense_type") + 1)
#     expense_unit_col = get_column_letter(header.index("expense_unit") + 1)

#     # Add data validation (dropdown) for expense_type
#     dv_type = DataValidation(type="list", formula1=f'"{",".join(expense_type_values)}"', allow_blank=True)
#     ws.add_data_validation(dv_type)
#     dv_type.add(f"{expense_type_col}2:{expense_type_col}1048576")

#     # Add data validation (dropdown) for expense_unit
#     dv_unit = DataValidation(type="list", formula1=f'"{",".join(expense_unit_values)}"', allow_blank=True)
#     ws.add_data_validation(dv_unit)
#     dv_unit.add(f"{expense_unit_col}2:{expense_unit_col}1048576")

#     # Add data validation (dropdown) for category
#     dv_cat = DataValidation(type="list", formula1=f'=lists!$A$1:$A${len(categories)}', allow_blank=True)
#     ws.add_data_validation(dv_cat)
#     dv_cat.add(f"{category_col}2:{category_col}1048576")

#     # Add data validation (dropdown) for currency iso code
#     dv_cur = DataValidation(type="list", formula1=f'=lists!$B$1:$B${len(currencies)}', allow_blank=True)
#     ws.add_data_validation(dv_cur)
#     dv_cur.add(f"{currency_col}2:{currency_col}1048576")

#     # Optionally hide the lists sheet
#     ws_lists.sheet_state = "hidden"

#     output = io.BytesIO()
#     wb.save(output)
#     output.seek(0)

#     return StreamingResponse(
#         output,
#         media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
#         headers={"Content-Disposition": 'attachment; filename="Expense Template.xlsx"'},
#     )


# @expense_router.post(
#     "/upload/",
#     dependencies=[Depends(authorize(expense_permission_mapper["upload_expense"]))],
#     status_code=status.HTTP_201_CREATED,
# )
# async def upload_expense(
#     expense_upload: Annotated[
#         schemas.ExpenseUpload, Form(media_type="multipart/form-data")
#     ],
#     request_user: User = Depends(get_request_user),
#     session: AsyncSession = Depends(get_async_session),
# ):
#     return await utils.handle_expense_upload(session, expense_upload, request_user)


# @expense_router.get(
#     "/download/",
#     dependencies=[Depends(authorize(expense_permission_mapper["download_expense"]))],
# )
# async def download_expense(
#     request_user: User = Depends(get_request_user),
#     session: AsyncSession = Depends(get_async_session),
# ):
#     buffer = await utils.handle_expense_download(session, request_user)
#     return StreamingResponse(
#         buffer,
#         media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
#         headers={"Content-Disposition": 'attachment; filename="Expenses.xlsx"'},
#     )
