import json
import base64
import importlib.resources as pkg_resources
from dateutil.relativedelta import relativedelta

import bcrypt
from celery import Task
from fastapi import HTTPException
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit import dateutils
from be_kit.caches import AsyncCache
from be_kit.cryptography import cipher
from be_kit.paginations import PaginationQuery
from be_kit.mailing import EmailClient
from ..settings import settings
from ..tasks import task_handler
from ..audit_log.enums import AuditLogAction
from ..audit_log.utils import audit_log_action, get_before_after
from . import enums, models, schemas, repositories


def get_password_hash(password: str):
    pwd_bytes = password.encode("utf-8")
    salt = bcrypt.gensalt()
    hashed_password = bcrypt.hashpw(password=pwd_bytes, salt=salt)
    return str(hashed_password, "utf-8")


def verify_password(password: str, hashed_password: str):
    password_byte_enc = password.encode("utf-8")
    hashed_password_byte_enc = hashed_password.encode("utf-8")
    return bcrypt.checkpw(
        password=password_byte_enc, hashed_password=hashed_password_byte_enc
    )


async def create_user(
    db: AsyncSession,
    cache: AsyncCache,
    user: schemas.UserCreate | schemas.InitialUserCreate,
    request_user: models.User = None,
    initial: bool = False,
):
    repo = (
        repositories.UserRepository(db)
        if not initial
        else repositories.InitialUserRepository(db)
    )
    user.password = get_password_hash(user.password)
    if initial:
        obj = await repo.acreate(
            **user.model_dump(), is_superuser=True, need_change_password=False
        )
    else:
        obj = await repo.acreate(
            **user.model_dump(),
            created_by_id=request_user.pk,
            last_modified_by_id=request_user.pk,
        )
        await audit_log_action(
            db,
            app="uam",
            module="user",
            submodule="user",
            action=AuditLogAction.CREATE,
            description=f"User {obj.standard_full_name} created.",
            created_by=request_user,
            before=None,
            after=None,
            _commit=True,
        )
    return obj


async def retrieve_user(db: AsyncSession, pk: int, request_user: models.User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.UserRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_users(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.UserFilter,
    ordering: list[str] | None,
    request_user: models.User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.UserRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_user(
    db: AsyncSession,
    pk: int,
    user: schemas.UserUpdate,
    request_user: models.User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.UserRepository(db, organization_id=group.organization_id)
    obj = await repo.aget(pk)
    old_obj = schemas.User.model_validate(obj)
    if obj.is_superuser and obj.group_id != user.group_id:
        raise HTTPException(400, "Can not update superuser group.")
    obj = await repo.aupdate(
        pk, **user.model_dump(), last_modified_by_id=request_user.pk
    )
    before, after = get_before_after(old_obj, schemas.User.model_validate(obj))
    await audit_log_action(
        db,
        app="uam",
        module="user",
        submodule="user",
        action=AuditLogAction.UPDATE,
        description=f"User {obj.standard_full_name} updated.",
        created_by=request_user,
        before=before,
        after=after,
        _commit=True,
    )
    return obj


async def delete_user(db: AsyncSession, pk: int, request_user: models.User = None):
    group = await request_user.awaitable_attrs.group
    repo = repositories.UserRepository(db, organization_id=group.organization_id)
    obj = await repo.aget_or_404(pk)
    await repo.aupdate(pk, is_active=False, last_modified_by_id=request_user.pk)
    await audit_log_action(
        db,
        app="uam",
        module="user",
        submodule="user",
        action=AuditLogAction.DELETE,
        description=f"User {obj.standard_full_name} deleted.",
        created_by=request_user,
        before=None,
        after=None,
        _commit=True,
    )


async def get_verification_token(scope: str, email: EmailStr) -> str:
    exp = dateutils.now() + relativedelta(minutes=10)
    payload = {
        "scope": scope,
        "email": email,
        "exp": exp.timestamp(),
    }
    token = cipher.encrypt(json.dumps(payload).encode())
    return base64.urlsafe_b64encode(token).decode()


async def decode_verification_token(token: str) -> dict:
    token = base64.urlsafe_b64decode(token)
    payload = json.loads(cipher.decrypt(token).decode())
    payload["exp"] = dateutils.from_timestamp(payload["exp"])
    if payload["exp"] < dateutils.now():
        raise HTTPException(400, "Expired token.")
    return payload


@task_handler.async_task(bind=False)
async def send_verification_email(
    context: str,
    email: EmailStr,
    last_name: str,
):
    email_client = EmailClient(
        settings=settings.mailing,
        template_dir=pkg_resources.files("be_uam.templates").joinpath("email"),
        test_mode=settings.test_mode,
        debug=settings.debug,
    )
    token = await get_verification_token(context, email)
    url = (
        settings.mailing.verifications_url.unicode_string() + f"?token={token}"
        if not settings.test_mode
        else f"http://localhost:8000/?token={token}"
    )
    message = await email_client.get_templated_message(
        template_name="verification.html",
        subject=f"Symbolix {context} Confirmation",
        to=email,
        body={
            "context": context,
            "full_name": last_name,
            "url": url,
            "year": dateutils.now().year,
        },
    )
    await email_client.send_message(message)


send_verification_email: Task


async def confirm_verification(db: AsyncSession, token: str) -> dict:
    try:
        payload = await decode_verification_token(token)
    except Exception as exc:
        raise HTTPException(400, "Invalid verification token.") from exc
    repo = repositories.AuthUserRepository(db)
    obj = await repo.aget_by_email(payload["email"])
    obj = await repo.aupdate(obj.pk, is_verified=True)
    return obj


async def set_user_password(
    db: AsyncSession,
    user_email: str,
    new_password: str,
    old_password: str | None = None,
):
    repo = repositories.AuthUserRepository(db)
    obj = await repo.aget_by_email(user_email)
    if old_password and not verify_password(old_password, obj.password):
        raise HTTPException(400, "Invalid old password.")
    new_password_hash = get_password_hash(new_password)
    obj = await repo.aupdate(
        obj.pk,
        password=new_password_hash,
        need_change_password=False,
    )
    await audit_log_action(
        db,
        app="uam",
        module="user",
        submodule="user",
        action=AuditLogAction.UPDATE,
        description=f"User {obj.standard_full_name} changed password.",
        created_by=obj,
        before=None,
        after=None,
        _commit=True,
    )
    return obj


async def change_user_password(
    db: AsyncSession,
    data: schemas.UserChangePassword,
):
    payload = await decode_verification_token(data.token)
    if payload["scope"] != enums.VerificationContext.CHANGE_PASSWORD.value:
        raise HTTPException(400, "Invalid verification token.")
    return await set_user_password(
        db, payload["email"], data.new_password, data.old_password
    )


async def reset_user_password(
    db: AsyncSession,
    data: schemas.UserResetPassword,
):
    payload = await decode_verification_token(data.token)
    if payload["scope"] not in [
        enums.VerificationContext.RESET_PASSWORD.value,
        enums.VerificationContext.NEW_USER.value,
    ]:
        raise HTTPException(400, "Invalid verification token.")
    return await set_user_password(db, payload["email"], data.new_password)
