import re
from datetime import datetime
from typing_extensions import Annotated

from pydantic import EmailStr, PositiveInt, StringConstraints, field_validator

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import BaseSchema, BaseORMSchema


class BaseUser:
    first_name: Annotated[str, StringConstraints(min_length=1, max_length=50)]
    last_name: Annotated[str, StringConstraints(min_length=1, max_length=50)]


class UsernameMixin:
    username: Annotated[str, StringConstraints(min_length=5, max_length=50)]


class EmailMixin:
    email: Annotated[EmailStr, StringConstraints(min_length=5, max_length=100)]


class PasswordMixin:
    password: Annotated[str, StringConstraints(min_length=8, max_length=50)]

    @field_validator("password", mode="after")
    @classmethod
    def password_complexity(cls, v: str):
        if not any(c.islower() for c in v):
            raise ValueError("Password must contain at least one lowercase letter")
        if not any(c.isupper() for c in v):
            raise ValueError("Password must contain at least one uppercase letter")
        if not any(c.isdigit() for c in v):
            raise ValueError("Password must contain at least one digit")
        if not re.search(r"[#$@!%&*?]", v):
            raise ValueError(
                "Password must contain at least one special character from [#$@!%&*?]"
            )
        return v


class GroupMixin:
    group_id: PositiveInt


class InitialUserCreate(PasswordMixin, EmailMixin, UsernameMixin, BaseUser, BaseSchema):
    pass


class UserCreate(
    PasswordMixin, GroupMixin, EmailMixin, UsernameMixin, BaseUser, BaseSchema
):
    pass


class UserUpdate(GroupMixin, BaseUser, BaseSchema):
    is_active: bool
    is_locked: bool


class User(EmailMixin, UsernameMixin, BaseUser, BaseORMSchema):
    is_superuser: bool
    is_active: bool
    is_verified: bool
    is_locked: bool
    last_login_at: datetime | None
    need_change_password: bool
    created_at: datetime
    last_modified_at: datetime
    created_by: "UserMin | None"
    last_modified_by: "UserMin | None"
    group: "UserGroupMin | None"


class PaginatedUser(BasePaginatedResponse):
    items: list[User]


class UserFilter(BaseSchema):
    username: str | None = None
    email: EmailStr | None = None
    first_name: EmailStr | None = None
    last_name: EmailStr | None = None
    is_superuser: bool | None = None
    is_active: bool | None = None
    is_verified: bool | None = None
    is_locked: bool | None = None
    group_id: bool | None = None
    username__icontains: str | None = None
    email__icontains: str | None = None
    first_name__icontains: str | None = None
    last_name__icontains: str | None = None
    group___name__icontains: str | None = None


class UserMin(EmailMixin, UsernameMixin, BaseUser, BaseORMSchema):
    pass


class UserOpt(UserMin):
    pass


class PaginatedUserOpt(BasePaginatedResponse):
    items: list[UserOpt]


class UserRegistrationVerification(BaseSchema):
    token: str


class UserChangePassword(BaseSchema):
    token: str
    old_password: str
    new_password: str


class UserResetPassword(BaseSchema):
    token: str
    new_password: str


# pylint: disable=wrong-import-position
from ..organization.schemas import OrganizationMin, GroupMin

# pylint: enable=wrong-import-position


class UserGroupMin(GroupMin):
    organization: OrganizationMin


class MetadataMixin:
    created_at: datetime
    last_modified_at: datetime
    created_by: "UserMin | None"
    last_modified_by: "UserMin | None"
