from datetime import datetime

from sqlalchemy import (
    BigInteger,
    Boolean,
    DateTime,
    ForeignKey,
    String,
    func,
    sql,
)
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.orm import Mapped, mapped_column, relationship

from be_kit import dateutils
from be_kit.databases import BaseModel
from ..organization.models import Group, Organization


class User(BaseModel):
    __tablename__ = "uam_user"

    pk: Mapped[int] = mapped_column(BigInteger, primary_key=True)
    username: Mapped[str] = mapped_column(String(50), unique=True, index=True)
    email: Mapped[str] = mapped_column(String(100), unique=True, index=True)
    password: Mapped[str] = mapped_column(String(128), default=None, nullable=True)
    first_name: Mapped[str] = mapped_column(String(50))
    last_name: Mapped[str] = mapped_column(String(50))
    is_superuser: Mapped[bool] = mapped_column(Boolean(), default=False)
    is_active: Mapped[bool] = mapped_column(Boolean(), default=True)
    is_verified: Mapped[bool] = mapped_column(Boolean(), server_default=sql.false())
    is_locked: Mapped[bool] = mapped_column(Boolean(), server_default=sql.false())
    last_login_at: Mapped[datetime | None] = mapped_column(
        DateTime(timezone=True), default=None, nullable=True
    )
    need_change_password: Mapped[bool] = mapped_column(
        Boolean(), server_default=sql.true()
    )
    # pylint: disable=not-callable
    created_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True), server_default=func.now()
    )
    last_modified_at: Mapped[datetime] = mapped_column(
        DateTime(timezone=True),
        server_default=func.now(),
        onupdate=dateutils.now,
    )
    # pylint: enable=not-callable
    created_by_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{__tablename__}.pk", ondelete="SET NULL"),
        default=None,
        nullable=True,
    )
    created_by: Mapped["User | None"] = relationship(
        "User",
        remote_side=[pk],
        foreign_keys=[created_by_id],
        post_update=True,
        passive_deletes=True,
        lazy="selectin",
        join_depth=1,
    )
    last_modified_by_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{__tablename__}.pk", ondelete="SET NULL"),
        default=None,
        nullable=True,
    )
    last_modified_by: Mapped["User | None"] = relationship(
        "User",
        remote_side=[pk],
        foreign_keys=[last_modified_by_id],
        post_update=True,
        passive_deletes=True,
        lazy="selectin",
        join_depth=1,
    )
    group_id: Mapped[int | None] = mapped_column(
        BigInteger, ForeignKey("uam_organization_group.pk", ondelete="RESTRICT")
    )
    group: Mapped[Group | None] = relationship(foreign_keys=[group_id], lazy="selectin")
    maintained_organization: Mapped[Organization | None] = relationship(
        back_populates="maintainer",
        foreign_keys=[Organization.maintainer_id],
        passive_deletes=True,
        lazy="selectin",
    )

    @property
    def standard_full_name(self) -> str:
        return f"{self.first_name} {self.last_name}"

    @property
    def reversed_full_name(self) -> str:
        return f"{self.last_name}, {self.first_name}"


class MetadataMixin:
    # pylint: disable=not-callable
    @declared_attr
    @classmethod
    def created_at(cls) -> Mapped[datetime]:
        return mapped_column(DateTime(timezone=True), server_default=func.now())

    @declared_attr
    @classmethod
    def last_modified_at(cls) -> Mapped[datetime]:
        return mapped_column(
            DateTime(timezone=True),
            server_default=func.now(),
            server_onupdate=func.now(),
        )

    @declared_attr
    @classmethod
    def created_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def created_by(cls) -> Mapped["User | None"]:
        return relationship(
            User,
            foreign_keys=[cls.created_by_id],
            passive_deletes=True,
            lazy="selectin",
        )

    @declared_attr
    @classmethod
    def last_modified_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def last_modified_by(cls) -> Mapped["User | None"]:
        return relationship(
            User,
            foreign_keys=[cls.last_modified_by_id],
            passive_deletes=True,
            lazy="selectin",
        )
