from typing import Annotated

from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.caches import AsyncCache
from be_kit.paginations import PaginationQuery
from ..auth.utils import get_request_user, authorize
from ..databases import get_async_session
from ..redis import get_async_cache
from . import enums, models, schemas, utils
from .permissions import permission_mapper
from fastapi import HTTPException

router = APIRouter(tags=["users"], prefix="/user")


@router.post(
    "/",
    dependencies=[Depends(authorize(permission_mapper["create_user"]))],
    response_model=schemas.User,
    status_code=status.HTTP_201_CREATED,
)
async def create_user(
    user: schemas.UserCreate,
    request_user: models.User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
    cache: AsyncCache = Depends(get_async_cache),
):
    obj = await utils.create_user(session, cache, user, request_user)
    utils.send_verification_email.delay(
        enums.VerificationContext.NEW_USER.value,
        obj.email,
        obj.last_name,
    )
    return obj


@router.post(
    "/initial/", response_model=schemas.User, status_code=status.HTTP_201_CREATED
)
async def create_initial_user(
    user: schemas.InitialUserCreate,
    session: AsyncSession = Depends(get_async_session),
    cache: AsyncCache = Depends(get_async_cache),
):
    obj = await utils.create_user(session, cache, user, initial=True)
    utils.send_verification_email.delay(
        enums.VerificationContext.REGISTRATION.value,
        obj.email,
        obj.last_name,
    )
    return obj


@router.get("/registration_verification/", response_model=schemas.User)
async def user_registration_verification(
    registration_verification: schemas.UserRegistrationVerification = Depends(),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.confirm_verification(session, registration_verification.token)


@router.get(
    "/{user_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["retrieve_user"]))],
    response_model=schemas.User,
)
async def retrieve_user(
    user_id: int,
    session: AsyncSession = Depends(get_async_session),
    request_user: models.User = Depends(get_request_user),
):
    return await utils.retrieve_user(session, user_id, request_user)


@router.get(
    "/",
    dependencies=[Depends(authorize(permission_mapper["list_user"]))],
    response_model=schemas.PaginatedUser,
)
async def list_user(
    pagination: PaginationQuery = Depends(),
    filters: schemas.UserFilter = Depends(),
    ordering: Annotated[list[enums.UserOrdering] | None, Query()] = None,
    request_user: models.User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_users(session, pagination, filters, ordering, request_user)


@router.get(
    "/options/",
    dependencies=[Depends(authorize(permission_mapper["list_user_option"]))],
    response_model=schemas.PaginatedUserOpt,
)
async def list_user_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.UserFilter = Depends(),
    ordering: Annotated[list[enums.UserOrdering] | None, Query()] = None,
    request_user: models.User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_users(session, pagination, filters, ordering, request_user)

@router.get(
    "/check-username/",
    status_code=200,
)
async def check_username_available(
    username: str = Query(..., min_length=3),
    session: AsyncSession = Depends(get_async_session),
):
    from .models import User
    from sqlalchemy import select
    result = await session.execute(select(User).where(User.username == username))
    user = result.scalar_one_or_none()
    if user:
        raise HTTPException(status_code=400, detail="Username is not available")
    return {"detail": "Username is available"}


@router.put(
    "/{user_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["update_user"]))],
    response_model=schemas.User,
)
async def update_user(
    user: schemas.UserUpdate,
    user_id: int,
    request_user: models.User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_user(session, user_id, user, request_user)


@router.put(
    "/change_password/",
    response_model=schemas.User,
)
async def change_user_password(
    data: schemas.UserChangePassword,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.change_user_password(session, data)


@router.put(
    "/reset_password/",
    response_model=schemas.User,
)
async def reset_user_password(
    data: schemas.UserResetPassword,
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.reset_user_password(session, data)


@router.delete(
    "/{user_id:int}/",
    dependencies=[Depends(authorize(permission_mapper["delete_user"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_user(
    user_id: int,
    request_user: models.User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_user(session, user_id, request_user)
