from sqlalchemy import BigInteger, ForeignKey, String, UniqueConstraint, func
from sqlalchemy.ext.hybrid import hybrid_property
from sqlalchemy.orm import Mapped, mapped_column

from be_kit.databases import BaseModel


class Permission(BaseModel):
    __tablename__ = "uam_permission"
    __table_args__ = (
        UniqueConstraint(
            "app",
            "module",
            "submodule",
            "action",
        ),
    )

    app: Mapped[str] = mapped_column(String(30))
    module: Mapped[str] = mapped_column(String(30))
    submodule: Mapped[str] = mapped_column(String(30))
    action: Mapped[str] = mapped_column(String(50))

    @hybrid_property
    def code(self) -> str:
        return f"{self.app}.{self.module}.{self.submodule}.{self.action}"

    # pylint: disable=not-callable
    @code.expression
    @classmethod
    def code(cls):
        return func.concat(
            cls.app, ".", cls.module, ".", cls.submodule, ".", cls.action
        )

    # pylint: enable=not-callable

    def __repr__(self) -> str:
        return f"<Permission {self.code}>"


class GroupPermission(BaseModel):
    __tablename__ = "uam_organization_group_permission"
    __table_args__ = (
        UniqueConstraint(
            "group_id",
            "permission_id",
        ),
    )

    group_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey("uam_organization_group.pk", ondelete="RESTRICT"),
    )
    permission_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{Permission.__tablename__}.pk", ondelete="RESTRICT"),
    )
