from fastapi import HTTPException
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.caches import AsyncCache
from be_kit.paginations import PaginationQuery
from ..user.models import User
from ..user.repositories import AuthUserRepository
from ..user.utils import get_password_hash
from ..permission.repositories import PermissionRepository
from ..audit_log.enums import AuditLogAction
from ..audit_log.utils import audit_log_action, get_before_after
from . import schemas, repositories


# Organization


async def create_initial_organization(
    db: AsyncSession,
    cache: AsyncCache,
    organization: schemas.InitialOrganizationCreate,
    request_user: User,
):
    user_repo = AuthUserRepository(db)
    request_user = await user_repo.aget(request_user.pk)
    repo = repositories.InitialOrganizationRepository(db)
    obj = await repo.acreate_and_setup(
        user=request_user,
        **organization.model_dump(),
        _commit=False,
    )
    await db.commit()
    obj = await repo.aget(obj.pk)
    return obj


async def create_child_organization(
    db: AsyncSession,
    child_organization: schemas.ChildOrganizationCreate,
    request_user: User,
):
    auth_user_repo = AuthUserRepository(db)
    request_user = await auth_user_repo.aget(request_user.pk)
    if not request_user.is_superuser:
        raise HTTPException(400, "Only superusers can create child organizations.")
    child_organization_data = child_organization.model_dump()
    initial_user_data = child_organization_data.pop("initial_user", None)
    initial_user_data["password"] = get_password_hash(initial_user_data["password"])
    repo = repositories.InitialOrganizationRepository(db)
    user_repo = repositories.InitialUserRepository(db)
    initial_user = await user_repo.acreate(
        **initial_user_data,
        is_superuser=False,
        need_change_password=True,
        created_by_id=request_user.pk,
        last_modified_by_id=request_user.pk,
        _commit=False,
    )
    obj = await repo.acreate_and_setup(
        user=initial_user,
        **child_organization_data,
        _commit=False,
    )
    await db.commit()
    obj = await repo.aget(obj.pk)
    return obj


async def retrieve_organization(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    if group.organization.parent_id is not None:
        repo = repositories.OrganizationRepository(
            db, organization_id=group.organization_id
        )
    else:
        repo = repositories.InitialOrganizationRepository(db)
    obj = await repo.aget_or_404(pk)
    return obj


async def list_organizations(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.OrganizationFilter,
    ordering: list[str] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    if group.organization.parent_id is not None:
        raise HTTPException(400, "Only root organizations can list organizations.")
    repo = repositories.InitialOrganizationRepository(db)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_organization(
    db: AsyncSession,
    pk: int,
    organization: schemas.OrganizationUpdate,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrganizationRepository(
        db, organization_id=group.organization_id
    )
    obj = await repo.aget_or_404(pk)
    old_obj = schemas.Organization.model_validate(obj)
    obj = await repo.aupdate(
        pk, **organization.model_dump(), last_modified_by_id=request_user.pk
    )
    before, after = get_before_after(old_obj, schemas.Organization.model_validate(obj))
    await audit_log_action(
        db,
        app="uam",
        module="organization",
        submodule="organization",
        action=AuditLogAction.UPDATE,
        description=f"Organization {obj.name} updated.",
        created_by=request_user,
        before=before,
        after=after,
        _commit=True,
    )
    return obj


async def delete_organization(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrganizationRepository(
        db, organization_id=group.organization_id
    )
    await repo.adelete(pk)


async def transfer_organization_maintainer(
    db: AsyncSession,
    pk: int,
    transfer_maintainer: schemas.TransferOrganizationMaintainer,
    request_user: User,
):
    if request_user.pk == transfer_maintainer.maintainer_id:
        raise HTTPException(400, "Cannot transfer to self.")
    group = await request_user.awaitable_attrs.group
    repo = repositories.OrganizationRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    if request_user.pk != obj.maintainer_id:
        raise HTTPException(
            400, "Transfer ownership is restricted to organization maintainer."
        )
    obj = await repo.atransfer_maintainer(
        pk, transfer_maintainer.maintainer_id, request_user
    )
    return obj


# Group


async def create_group(
    db: AsyncSession,
    group: schemas.GroupCreate,
    request_user: User,
):
    repo = repositories.GroupRepository(db)
    perm_repo = PermissionRepository(db)
    user_group = await request_user.awaitable_attrs.group
    group_data = group.model_dump()
    permission_ids = group_data.pop("permission_ids")
    permissions = await perm_repo.aget_by_ids(permission_ids)
    group_data["permissions"] = list(permissions)
    obj = await repo.acreate(
        **group_data,
        organization_id=user_group.organization_id,
        created_by_id=request_user.pk,
    )
    await audit_log_action(
        db,
        app="uam",
        module="organization",
        submodule="group",
        action=AuditLogAction.CREATE,
        description=f"Group {obj.name} created.",
        created_by=request_user,
        before=None,
        after=None,
        _commit=True,
    )
    return obj


async def retrieve_group(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.GroupRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_groups(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.GroupFilter,
    ordering: list[str] | None,
    request_user: User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.GroupRepository(
        db,
        organization_id=group.organization_id,
    )
    objs = await repo.arecords(filters, pagination, ordering)
    return objs


async def update_group(
    db: AsyncSession,
    pk: int,
    group: schemas.GroupUpdate,
    request_user: User,
):
    user_group = await request_user.awaitable_attrs.group
    repo = repositories.GroupRepository(
        db,
        organization_id=user_group.organization_id,
    )
    perm_repo = PermissionRepository(db)
    obj = await repo.aget(pk)
    if obj.parent_id is None and obj.parent_id != group.parent_id:
        raise HTTPException(400, "Can not update root group parent.")
    old_obj = schemas.Group.model_validate(obj)
    group_data = group.model_dump()
    permission_ids = group_data.pop("permission_ids")
    permissions = await perm_repo.aget_by_ids(permission_ids)
    group_data["permissions"] = list(permissions)
    obj = await repo.aupdate(pk, **group_data, last_modified_by_id=request_user.pk)
    before, after = get_before_after(old_obj, schemas.Group.model_validate(obj))
    await audit_log_action(
        db,
        app="uam",
        module="organization",
        submodule="group",
        action=AuditLogAction.UPDATE,
        description=f"Group {obj.name} updated.",
        created_by=request_user,
        before=before,
        after=after,
        _commit=True,
    )
    return obj


async def delete_group(db: AsyncSession, pk: int, request_user: User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.GroupRepository(db, organization_id=group.organization_id)
    obj = await repo.aget(pk)
    if not obj.parent_id:
        raise HTTPException(400, "Can not delete root group.")
    all_groups = await repo.arecords(schemas.GroupFilter(), PaginationQuery(limit=10000), None)
    has_child = any(g.parent_id == pk for g in all_groups['items'])
    if has_child:
        raise HTTPException(400, "Can not delete group with child groups. Remove or reassign child groups first.")
    await repo.adelete(pk)
    await audit_log_action(
        db,
        app="uam",
        module="organization",
        submodule="group",
        action=AuditLogAction.DELETE,
        description=f"Group {obj.name} deleted.",
        created_by=request_user,
        before=None,
        after=None,
        _commit=True,
    )
