from datetime import datetime
from typing_extensions import Annotated

from pydantic import PositiveInt, StringConstraints

from be_kit.paginations import BasePaginatedResponse
from be_kit.schemas import (
    BaseSchema,
    BaseORMSchema,
    SoftDeleteMixin,
    SoftDeleteFilterMixin,
)
from ..permission.schemas import Permission
from .enums import OrganizationLegalType


class MetadataMixin:
    created_at: datetime
    last_modified_at: datetime
    created_by: "UserMin | None"
    last_modified_by: "UserMin | None"


# Organization


class BaseOrganization:
    legal_type: OrganizationLegalType
    legal_name: Annotated[str, StringConstraints(min_length=3, max_length=150)]


class InitialOrganizationCreate(BaseOrganization, BaseSchema):
    pass


class ChildOrganizationCreate(BaseOrganization, BaseSchema):
    parent_id: PositiveInt | None = None
    initial_user: "InitialUserCreate"


class OrganizationUpdate(BaseOrganization, BaseSchema):
    pass


class Organization(SoftDeleteMixin, MetadataMixin, BaseOrganization, BaseORMSchema):
    maintainer: "UserMin"
    parent: "OrganizationMin | None"


class PaginatedOrganization(BasePaginatedResponse):
    items: list[Organization]


class OrganizationFilter(SoftDeleteFilterMixin, BaseSchema):
    legal_type: OrganizationLegalType | None = None
    parent_id: PositiveInt | None = None


class OrganizationMin(BaseOrganization, BaseORMSchema):
    parent: "OrganizationParentMin | None"


class OrganizationOpt(BaseOrganization, BaseORMSchema):
    pass


class PaginatedOrganizationOpt(BasePaginatedResponse):
    items: list[OrganizationOpt]


class OrganizationParentMin(BaseOrganization, BaseORMSchema):
    pass


class TransferOrganizationMaintainer(BaseSchema):
    maintainer_id: PositiveInt


class OrganizationMixin:
    organization: OrganizationMin


# Group


class BaseGroup:
    name: Annotated[str, StringConstraints(min_length=3, max_length=150)]


class ParentIdMixin:
    parent_id: PositiveInt | None


class PermissionIdsMixin:
    permission_ids: list[PositiveInt]


class PermissionMixin:
    permissions: list[Permission]


class GroupCreate(PermissionIdsMixin, ParentIdMixin, BaseGroup, BaseSchema):
    pass


class GroupUpdate(PermissionIdsMixin, ParentIdMixin, BaseGroup, BaseSchema):
    pass


class Group(SoftDeleteMixin, MetadataMixin, PermissionMixin, BaseGroup, BaseORMSchema):
    organization: OrganizationMin
    parent: "GroupMin | None"


class GroupList(SoftDeleteMixin, MetadataMixin, BaseGroup, BaseORMSchema):
    organization: OrganizationMin
    parent: "GroupMin | None"


class PaginatedGroup(BasePaginatedResponse):
    items: list[GroupList]


class GroupFilter(SoftDeleteFilterMixin, BaseSchema):
    name: str | None = None
    parent_id: PositiveInt | None = None
    name__icontains: str | None = None


class GroupMin(BaseGroup, BaseORMSchema):
    pass


class GroupOpt(GroupMin):
    pass


class PaginatedGroupOpt(BasePaginatedResponse):
    items: list[GroupOpt]


class GroupPermission(BaseORMSchema):
    permissions: list[Permission]


# pylint: disable=wrong-import-position
from ..user.schemas import UserMin, InitialUserCreate

# pylint: enable=wrong-import-position


UserMin.model_rebuild()
