from sqlalchemy import Select

from be_kit import dateutils
from be_kit.repositories import BaseRepository

from ..permission.repositories import PermissionRepository
from ..user.models import User
from ..user.repositories import InitialUserRepository, UserRepository
from . import models


class BaseOrganizationMixin:
    model = models.Organization


class OrganizationRepository(BaseOrganizationMixin, BaseRepository):
    class_default_filters = {"deleted_at": None}
    filter_by_organization = True

    def add_organization_filter(self, query: Select):
        query = query.filter_by(pk=self.organization_id)
        return query

    async def adelete(self, pk):
        obj = await self.aget_or_404(pk)
        obj = self.update_values(obj, {"deleted_at": dateutils.now()})
        try:
            self.session.add(obj)
            await self.session.commit()
        except:
            await self.session.rollback()
            raise
        user_repo = UserRepository(self.session, organization_id=obj.pk)
        for user in await user_repo.arecords():
            if user.pk == obj.maintainer_id:
                continue
            await user_repo.aupdate(
                user.pk, is_active=False, last_modified_by_id=obj.maintainer_id
            )

    async def atransfer_maintainer(self, pk, maintainer_id: int, request_user: User):
        obj = await self.aupdate(
            pk, maintainer_id=maintainer_id, last_modified_by_id=request_user.pk
        )
        root_organization = obj.parent_id is None
        user_repo = UserRepository(self.session, organization_id=obj.pk)
        await user_repo.aupdate(
            maintainer_id,
            is_superuser=root_organization,
            group_id=request_user.group_id,
            last_modified_by_id=request_user.pk,
            _commit=False,
        )
        await user_repo.aupdate(
            request_user.pk,
            is_superuser=False,
            last_modified_by_id=request_user.pk,
            _commit=False,
        )
        await self.session.commit()
        return await self.aget(pk)


class InitialOrganizationRepository(BaseOrganizationMixin, BaseRepository):
    async def acreate_and_setup(self, user: User, **kwargs):
        _commit = kwargs.pop("_commit", True)
        group_repo = InitialGroupRepository(self.session)
        perm_repo = PermissionRepository(self.session)
        user_repo = InitialUserRepository(self.session)
        obj = await super().acreate(
            maintainer_id=user.pk, created_by_id=user.pk, **kwargs, _commit=_commit
        )
        group = await group_repo.acreate(
            name="Root",
            organization_id=obj.pk,
            created_by_id=user.pk,
            _commit=_commit,
        )
        default_perms = await perm_repo.arecords()
        default_perms = list(default_perms)
        group = await group_repo.aupdate(
            group.pk, permissions=default_perms, _commit=_commit
        )
        await user_repo.aupdate(user.pk, group_id=group.pk, _commit=_commit)
        return await self.aget(obj.pk)


class BaseGroupMixin:
    model = models.Group


class GroupRepository(BaseGroupMixin, BaseRepository):
    filter_by_organization = True

    def add_organization_filter(self, query: Select):
        query = query.filter_by(organization_id=self.organization_id)
        return query


class InitialGroupRepository(BaseGroupMixin, BaseRepository):
    pass


class GroupTaskRepository(GroupRepository):
    filter_by_organization = False
