from datetime import datetime
from typing import TYPE_CHECKING

from sqlalchemy import (
    BigInteger,
    DateTime,
    Enum,
    ForeignKey,
    String,
    UniqueConstraint,
    func,
)
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.orm import Mapped, mapped_column, relationship


from be_kit.databases import BaseModel, SoftDeleteMixin
from ..utils import get_user_cls
from ..permission.models import Permission, GroupPermission
from .enums import OrganizationLegalType

if TYPE_CHECKING:
    from ..user.models import User


class MetadataMixin:
    # pylint: disable=not-callable
    @declared_attr
    @classmethod
    def created_at(cls) -> Mapped[datetime]:
        return mapped_column(DateTime(timezone=True), server_default=func.now())

    @declared_attr
    @classmethod
    def last_modified_at(cls) -> Mapped[datetime]:
        return mapped_column(
            DateTime(timezone=True),
            server_default=func.now(),
            server_onupdate=func.now(),
        )

    @declared_attr
    @classmethod
    def created_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def created_by(cls) -> Mapped["User | None"]:
        return relationship(
            "User",
            foreign_keys=[cls.created_by_id],
            passive_deletes=True,
            lazy="selectin",
        )

    @declared_attr
    @classmethod
    def last_modified_by_id(cls) -> Mapped[int | None]:
        return mapped_column(
            BigInteger,
            ForeignKey("uam_user.pk", ondelete="SET NULL"),
            default=None,
            nullable=True,
        )

    @declared_attr
    @classmethod
    def last_modified_by(cls) -> Mapped["User | None"]:
        return relationship(
            "User",
            foreign_keys=[cls.last_modified_by_id],
            passive_deletes=True,
            lazy="selectin",
        )


class Organization(SoftDeleteMixin, MetadataMixin, BaseModel):
    __tablename__ = "uam_organization"

    # pylint: disable=unsubscriptable-object
    pk: Mapped[int] = mapped_column(BigInteger, primary_key=True)
    legal_type: Mapped[OrganizationLegalType] = mapped_column(
        Enum(OrganizationLegalType), index=True
    )
    legal_name: Mapped[str] = mapped_column(String(150), unique=True, index=True)
    maintainer_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey("uam_user.pk", ondelete="RESTRICT")
    )
    maintainer: Mapped["User"] = relationship(
        get_user_cls,
        back_populates="maintained_organization",
        foreign_keys=[maintainer_id],
        passive_deletes=True,
        lazy="selectin",
    )
    parent_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
    )
    parent: Mapped["Organization | None"] = relationship(
        "be_uam.organization.models.Organization",
        remote_side=[pk],
        foreign_keys=[parent_id],
        passive_deletes=True,
        join_depth=1,
        lazy="selectin",
    )
    # pylint: enable=unsubscriptable-object


class OrganizationMixin:
    @declared_attr
    @classmethod
    def organization_id(cls) -> Mapped[int]:
        return mapped_column(
            BigInteger,
            ForeignKey(f"{Organization.__tablename__}.pk", ondelete="RESTRICT"),
        )

    @declared_attr
    @classmethod
    def organization(cls):
        return relationship(
            Organization,
            foreign_keys=[cls.organization_id],
            passive_deletes=True,
            lazy="selectin",
        )


class Group(SoftDeleteMixin, MetadataMixin, BaseModel):
    __tablename__ = "uam_organization_group"
    __table_args__ = (UniqueConstraint("organization_id", "name"),)

    # pylint: disable=unsubscriptable-object
    pk: Mapped[int] = mapped_column(BigInteger, primary_key=True)
    organization_id: Mapped[int] = mapped_column(
        BigInteger, ForeignKey(f"{Organization.__tablename__}.pk", ondelete="RESTRICT")
    )
    organization: Mapped[Organization] = relationship(
        Organization, passive_deletes=True, lazy="selectin"
    )
    name: Mapped[str] = mapped_column(String(150))
    parent_id: Mapped[int | None] = mapped_column(
        BigInteger,
        ForeignKey(f"{__tablename__}.pk", ondelete="RESTRICT"),
        nullable=True,
    )
    parent: Mapped["Group | None"] = relationship(
        "be_uam.organization.models.Group",
        remote_side=[pk],
        foreign_keys=[parent_id],
        passive_deletes=True,
        join_depth=1,
        lazy="selectin",
    )
    permissions: Mapped[list[Permission]] = relationship(
        secondary=GroupPermission.__tablename__,
        passive_deletes=True,
        lazy="selectin",
    )
    # pylint: enable=unsubscriptable-object
