from typing import Annotated
from fastapi import APIRouter, Depends, Query, status
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.caches import AsyncCache
from be_kit.paginations import PaginationQuery
from ..databases import get_async_session
from ..auth.utils import get_request_user, authorize
from ..redis import get_async_cache
from ..user.models import User
from . import enums, schemas, utils
from .permissions import (
    organization_permission_mapper,
    group_permission_mapper,
)


# Organization

organization_router = APIRouter(tags=["organizations"], prefix="/organization")


@organization_router.post(
    "/initial/",
    response_model=schemas.Organization,
    status_code=status.HTTP_201_CREATED,
)
async def create_initial_organization(
    organization: schemas.InitialOrganizationCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
    cache: AsyncCache = Depends(get_async_cache),
):
    return await utils.create_initial_organization(
        session, cache, organization, request_user
    )


@organization_router.post(
    "/child/",
    response_model=schemas.Organization,
    status_code=status.HTTP_201_CREATED,
)
async def create_child_organization(
    organization: schemas.ChildOrganizationCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_child_organization(session, organization, request_user)


@organization_router.get(
    "/{organization_id:int}/",
    dependencies=[
        Depends(authorize(organization_permission_mapper["retrieve_organization"]))
    ],
    response_model=schemas.Organization,
)
async def retrieve_organization(
    organization_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_organization(session, organization_id, request_user)


@organization_router.get(
    "/",
    dependencies=[
        Depends(authorize(organization_permission_mapper["list_organization"]))
    ],
    response_model=schemas.PaginatedOrganization,
)
async def list_organization(
    pagination: PaginationQuery = Depends(),
    filters: schemas.OrganizationFilter = Depends(),
    ordering: Annotated[list[enums.OrganizationOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_organizations(
        session, pagination, filters, ordering, request_user
    )


@organization_router.get(
    "/options/",
    dependencies=[
        Depends(authorize(organization_permission_mapper["list_organization_option"]))
    ],
    response_model=schemas.PaginatedOrganizationOpt,
)
async def list_organization_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.OrganizationFilter = Depends(),
    ordering: Annotated[list[enums.OrganizationOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_organizations(
        session, pagination, filters, ordering, request_user
    )


@organization_router.put(
    "/{organization_id:int}/",
    dependencies=[
        Depends(authorize(organization_permission_mapper["update_organization"]))
    ],
    response_model=schemas.Organization,
)
async def update_organization(
    organization: schemas.OrganizationUpdate,
    organization_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_organization(
        session, organization_id, organization, request_user
    )


@organization_router.delete(
    "/{organization_id:int}/",
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_organization(
    organization_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_organization(session, organization_id, request_user)


@organization_router.put(
    "/{organization_id:int}/transfer_maintainer/",
    response_model=schemas.Organization,
)
async def transfer_organization_maintainer(
    transfer_maintainer: schemas.TransferOrganizationMaintainer,
    organization_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.transfer_organization_maintainer(
        session, organization_id, transfer_maintainer, request_user
    )


# Group

group_router = APIRouter(tags=["groups"], prefix="/group")


@group_router.post(
    "/",
    dependencies=[Depends(authorize(group_permission_mapper["create_group"]))],
    response_model=schemas.Group,
    status_code=status.HTTP_201_CREATED,
)
async def create_group(
    group: schemas.GroupCreate,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.create_group(session, group, request_user)


@group_router.get(
    "/{group_id:int}/",
    dependencies=[Depends(authorize(group_permission_mapper["retrieve_group"]))],
    response_model=schemas.Group,
)
async def retrieve_group(
    group_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_group(session, group_id, request_user)


@group_router.get(
    "/",
    dependencies=[Depends(authorize(group_permission_mapper["list_group"]))],
    response_model=schemas.PaginatedGroup,
)
async def list_group(
    pagination: PaginationQuery = Depends(),
    filters: schemas.GroupFilter = Depends(),
    ordering: Annotated[list[enums.GroupOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_groups(session, pagination, filters, ordering, request_user)


@group_router.get(
    "/options/",
    dependencies=[Depends(authorize(group_permission_mapper["list_group_option"]))],
    response_model=schemas.PaginatedGroupOpt,
)
async def list_group_option(
    pagination: PaginationQuery = Depends(),
    filters: schemas.GroupFilter = Depends(),
    ordering: Annotated[list[enums.GroupOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_groups(session, pagination, filters, ordering, request_user)


@group_router.put(
    "/{group_id:int}/",
    dependencies=[Depends(authorize(group_permission_mapper["update_group"]))],
    response_model=schemas.Group,
)
async def update_group(
    group: schemas.GroupUpdate,
    group_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.update_group(session, group_id, group, request_user)


@group_router.delete(
    "/{group_id:int}/",
    dependencies=[Depends(authorize(group_permission_mapper["delete_group"]))],
    status_code=status.HTTP_204_NO_CONTENT,
)
async def delete_group(
    group_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    await utils.delete_group(session, group_id, request_user)
