from datetime import datetime
from typing import Annotated

from pydantic import (
    BaseModel,
    EmailStr,
    StringConstraints,
    computed_field,
    field_validator,
)

from be_kit.schemas import BaseSchema
from ..redis import cache_config
from ..user.schemas import User
from ..user.enums import VerificationContext
from .enums import AuthorizationMode


class Token(BaseSchema):
    access_token: str
    refresh_token: str
    token_type: str


class LoginForm(BaseModel):
    username: str
    password: str


class AuthUser(User):
    @computed_field
    @property
    def otp_verified(self) -> bool:
        cache = cache_config.get_sync_cache()
        status = cache.get("uam", "auth", "otp_verified", self.email)
        cache.close()
        return status if status is not None else False


class OTPVerification(BaseSchema):
    otp: Annotated[str, StringConstraints(min_length=6, max_length=6)]


class PermissionCheck(BaseSchema):
    actions: str
    mode: AuthorizationMode


class PermissionCheckFilter(BaseSchema):
    codes: str
    mode: AuthorizationMode = AuthorizationMode.ALL

    @field_validator("codes", mode="after")
    @classmethod
    def validate_codes(cls, v: str) -> list[str]:
        return v.split(",")


class VerificationToken(BaseSchema):
    token: str


class VerificationTokenPayload(BaseSchema):
    scope: VerificationContext
    email: EmailStr
    exp: datetime


class SetupState(BaseSchema):
    user: bool
    organization: bool
