from fastapi import (
    APIRouter,
    Depends,
    HTTPException,
    Header,
    Response,
    status,
)
from fastapi.security import OAuth2PasswordRequestForm
from pydantic import EmailStr
from sqlalchemy.ext.asyncio import AsyncSession


from be_kit.caches import AsyncCache
from be_kit.exceptions import HTTPUnauthorizedException
from ..databases import get_async_session
from ..permission.exceptions import InvalidPermissionActionError
from ..redis import get_async_cache
from ..settings import settings
from ..user.enums import VerificationContext
from ..user.models import User as UserModel
from ..user.schemas import User
from ..user.utils import send_verification_email, decode_verification_token
from ..organization.schemas import GroupPermission
from ..organization.utils import retrieve_group
from . import schemas, utils


router = APIRouter(tags=["auth"], prefix="/auth")


@router.post("/login/", response_model=schemas.Token)
async def login(
    response: Response,
    form_data: OAuth2PasswordRequestForm = Depends(),
    session: AsyncSession = Depends(get_async_session),
    cache: AsyncCache = Depends(get_async_cache),
):
    obj = await utils.authenticate(
        session, cache, username=form_data.username, password=form_data.password
    )
    access_token = await utils.create_access_token(data={"sub": obj.pk})
    refresh_token = await utils.create_refresh_token(data={"sub": obj.pk})
    response.set_cookie(
        key=settings.jwt.access_cookie_name,
        value=access_token,
        httponly=True,
        max_age=settings.jwt.access_token_expire * 24 * 60 * 60,
        secure=settings.ssl,
    )
    response.set_cookie(
        key=settings.jwt.refresh_cookie_name,
        value=refresh_token,
        httponly=True,
        max_age=settings.jwt.access_token_expire * 24 * 60 * 60,
        secure=settings.ssl,
    )
    utils.send_otp_email.delay(
        obj.email,
        obj.last_name,
    )

    return schemas.Token(
        access_token=access_token, refresh_token=refresh_token, token_type="bearer"
    )


@router.post("/logout/", status_code=status.HTTP_204_NO_CONTENT)
async def logout(response: Response):
    response.delete_cookie(key=settings.jwt.access_cookie_name)
    response.delete_cookie(key=settings.jwt.refresh_cookie_name)
    return


@router.post("/refresh/", response_model=schemas.Token)
async def refresh(
    response: Response,
    refresh_token: str | None = Header(None),
    session: AsyncSession = Depends(get_async_session),
):
    if not refresh_token:
        refresh_token = response.cookies.get(settings.jwt.refresh_cookie_name, None)
    if not refresh_token:
        raise HTTPUnauthorizedException

    obj = await utils.verify_token(session, refresh_token)
    access_token = await utils.create_access_token(data={"sub": obj.pk})

    response.set_cookie(
        key=settings.jwt.access_cookie_name,
        value=access_token,
        httponly=True,
        max_age=settings.jwt.access_token_expire * 24 * 60 * 60,
        secure=settings.ssl,
    )

    return schemas.Token(
        access_token=access_token, refresh_token=refresh_token, token_type="bearer"
    )


@router.get(
    "/profile/",
    response_model=schemas.AuthUser,
)
async def profile(
    request_user: UserModel = Depends(utils.get_request_user),
):
    return request_user


@router.get(
    "/verification_email/",
    status_code=status.HTTP_204_NO_CONTENT,
)
async def verification_email(
    context: VerificationContext,
    email: EmailStr | None = None,
    request_user: UserModel = Depends(utils.get_request_user_or_anon),
):
    if not request_user and not email:
        raise HTTPException(400, "Email is required.")
    send_verification_email.delay(
        context.value,
        request_user.email if request_user else email,
        request_user.last_name if request_user else email.split("@")[0],
    )


@router.get(
    "/verification_data/",
    response_model=schemas.VerificationTokenPayload,
)
async def verification_data(
    verification_token: schemas.VerificationToken = Depends(),
):
    return await decode_verification_token(verification_token.token)


@router.post(
    "/otp_verification/",
    status_code=status.HTTP_204_NO_CONTENT,
)
async def otp_verification(
    request_user: UserModel = Depends(utils.get_request_user),
    otp_verification_form: schemas.OTPVerification = Depends(),
    cache: AsyncCache = Depends(get_async_cache),
):
    result = await utils.verify_otp(cache, request_user.email, otp_verification_form.otp)
    if not result:
        raise HTTPException(status.HTTP_401_UNAUTHORIZED, "Invalid OTP.")
    return


@router.get(
    "/permission_check/",
    status_code=status.HTTP_204_NO_CONTENT,
)
async def permission_check(
    request_user: UserModel = Depends(utils.get_request_user),
    filters: schemas.PermissionCheckFilter = Depends(),
    session: AsyncSession = Depends(get_async_session),
    cache: AsyncCache = Depends(get_async_cache),
):
    try:
        await utils.check_permission(session, cache, request_user, filters.codes, filters.mode)
    except InvalidPermissionActionError as exc:
        raise HTTPException(400, str(exc)) from exc


@router.get(
    "/group_permissions/",
    response_model=GroupPermission,
)
async def get_permissions(
    request_user: UserModel = Depends(utils.get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await retrieve_group(session, request_user.group_id, request_user)

@router.post(
    "/resend_otp/",
    status_code=status.HTTP_204_NO_CONTENT,
)
async def resend_otp(
    request_user: UserModel = Depends(utils.get_request_user),
):
    utils.send_otp_email.delay(
        request_user.email,
        request_user.last_name,
    )
    return
