from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.schemas import BaseORMSchema
from be_kit.paginations import PaginationQuery
from . import enums, models, schemas, repositories
import datetime

async def audit_log_action(
    db: AsyncSession,
    app: str,
    module: str,
    submodule: str,
    action: enums.AuditLogAction,
    description: str,
    created_by: models.User,
    before: dict | None = None,
    after: dict | None = None,
    _commit: bool = True,
):
    repo = repositories.AuditLogRepository(db)
    group = await created_by.awaitable_attrs.group
    obj = await repo.acreate(
        app=app,
        module=module,
        submodule=submodule,
        action=action,
        description=description,
        created_by_id=created_by.pk,
        before=before,
        after=after,
        organization_id=group.organization_id,
        _commit=_commit,
    )
    return obj

def serialize_dict(obj):
    for k, v in obj.items():
        if isinstance(v, datetime.datetime):
            obj[k] = v.isoformat()
    return obj

def get_before_after(old: BaseORMSchema, new: BaseORMSchema):
    before = serialize_dict(old.model_dump())
    after = serialize_dict(new.model_dump())
    for key in list(before.keys()):
        if before[key] == after[key]:
            before.pop(key)
            after.pop(key)
    return before, after


async def retrieve_audit_log(db: AsyncSession, pk: int, request_user: models.User):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AuditLogRepository(
        db,
        organization_id=group.organization_id,
    )
    obj = await repo.aget_or_404(pk)
    return obj


async def list_audit_logs(
    db: AsyncSession,
    pagination: PaginationQuery,
    filters: schemas.AuditLogFilter,
    ordering: list[str] | None,
    request_user: models.User,
):
    group = await request_user.awaitable_attrs.group
    repo = repositories.AuditLogRepository(db, organization_id=group.organization_id)
    objs = await repo.arecords(filters, pagination, ordering)
    return objs
