from typing import Annotated
from fastapi import APIRouter, Depends, Query
from sqlalchemy.ext.asyncio import AsyncSession

from be_kit.paginations import PaginationQuery
from ..databases import get_async_session
from ..auth.utils import get_request_user
from ..user.models import User
from . import enums, schemas, utils


audit_log_router = APIRouter(tags=["audit_logs"], prefix="/audit_log")


@audit_log_router.get(
    "/{audit_log_id:int}/",
    response_model=schemas.AuditLog,
)
async def retrieve_audit_log(
    audit_log_id: int,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.retrieve_audit_log(session, audit_log_id, request_user)


@audit_log_router.get(
    "/",
    response_model=schemas.PaginatedAuditLog,
)
async def list_audit_log(
    pagination: PaginationQuery = Depends(),
    filters: schemas.AuditLogFilter = Depends(),
    ordering: Annotated[list[enums.AuditLogOrdering] | None, Query()] = None,
    request_user: User = Depends(get_request_user),
    session: AsyncSession = Depends(get_async_session),
):
    return await utils.list_audit_logs(
        session, pagination, filters, ordering, request_user
    )
